function parseVersion(input) {
    const data = {
        version: input.toLowerCase()
    };
    const parts = input.toLowerCase().split('.').filter((p)=>!!p);
    // Only parse single parts that are actual numbers.
    // This check will prevent `parseInt` from pasing the leading chars if they are
    // valid numbers.
    if (parts.length <= 1 && !/^\d+$/.test(parts[0])) {
        return data;
    }
    const major = parseInt(parts[0], 10);
    const minor = parseInt(parts[1], 10);
    const patch = parseInt(parts[2], 10);
    // Only add converted versions if `major` part was valid
    if (!Number.isNaN(major)) {
        data.major = major;
        if (!Number.isNaN(minor)) data.minor = minor;
        if (!Number.isNaN(patch)) data.patch = patch;
    }
    return data;
}
/**
 * Check if the given value is a complete Version struct.
 */ // eslint-disable-next-line @typescript-eslint/no-explicit-any
function isVersion(value) {
    var _value, _value1;
    return typeof ((_value = value) === null || _value === void 0 ? void 0 : _value.major) === 'number' && typeof ((_value1 = value) === null || _value1 === void 0 ? void 0 : _value1.minor) === 'number';
}
/**
 * Compare two version numbers together.
 *
 * NOTE: This only supports the first 3 segments (major, minor, patch) and does not
 *       do a full SemVer compare.
 *
 * @example
 * ```javascript
 * compareVersion('1.2.3', '1.2.4');
 * // => -1
 * ```
 */ function compareVersion(base, comp) {
    let baseList = toNumbers(base);
    let compList = toNumbers(comp);
    // Right pad versions with zeros to make them equal length
    const versionLength = Math.max(baseList.length, compList.length);
    baseList = baseList.concat(Array(versionLength).fill(0)).slice(0, versionLength);
    compList = compList.concat(Array(versionLength).fill(0)).slice(0, versionLength);
    /** Constrain the given value to the output range of this function. */ const constrain = (value)=>{
        if (value <= -1) return -1;
        else if (value >= 1) return 1;
        else return 0;
    };
    for(let index = 0; index < versionLength; index++){
        const aValue = baseList[index];
        const bValue = compList[index];
        if (aValue !== bValue) {
            return constrain(aValue - bValue);
        }
    }
    return 0;
}
function eq(base, comp) {
    return compareVersion(base, comp) === 0;
}
function gt(base, comp) {
    return compareVersion(base, comp) > 0;
}
function gte(base, comp) {
    const result = compareVersion(base, comp);
    return result > 0 || result === 0;
}
function lt(base, comp) {
    return compareVersion(base, comp) < 0;
}
function lte(base, comp) {
    const result = compareVersion(base, comp);
    return result < 0 || result === 0;
}
function toNumbers(value) {
    if (Array.isArray(value)) {
        return value;
    } else if (typeof value === 'number') {
        return [
            value
        ];
    } else if (typeof value === 'string') {
        return toNumbers(parseVersion(value));
    } else {
        const values = [
            value.major,
            value.minor,
            value.patch
        ];
        const uidx = values.indexOf(undefined);
        return uidx === -1 ? values : values.slice(0, uidx);
    }
}

export { compareVersion, eq, gt, gte, isVersion, lt, lte, parseVersion };
