"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LocalizedStringsBundle = void 0;
const environment_1 = require("@jet/environment");
const localized_strings_json_object_1 = require("./localized-strings-json-object");
/**
 * A localized string data source which loads strings from the application bundle.
 *
 * The bundle used by this data source can be a web app webpack bundle
 * or a native app bundle bridged over to JS code.
 */
class LocalizedStringsBundle {
    // MARK: - Initialization
    /**
     * Create localized strings bundle with all required attributes.
     *
     * @param bundle - The app bundle object.
     */
    constructor(bundle) {
        this.bundle = bundle;
    }
    // MARK: - LocalizedStringsDataSource
    async fetchStrings(language) {
        var _a;
        // Load the strings from bundle and cache them.
        const localizations = this.bundle.localizationsProperty;
        if (environment_1.isNothing(localizations)) {
            throw new Error("Localized strings bundle index file is missing 'localizations' property");
        }
        let strings;
        const format = (_a = localizations.format) !== null && _a !== void 0 ? _a : "json/inline" /* jsonInline */;
        if (format === "json/inline" /* jsonInline */) {
            const inlineLocalizations = localizations;
            strings = inlineLocalizations[language];
        }
        else {
            const externalLocalizations = localizations;
            switch (externalLocalizations.format) {
                case "json/multi-file" /* jsonMultiFile */:
                    {
                        // The path points to directory where JSON files are located.
                        // We don't even have to list a directory, just construct a final path.
                        // The path is also not an OS path but a bundle (e.g. JetPack) path.
                        // Bundle APIs always use "/" in the path, same as the paths used in the
                        // index.json (manifest) files.
                        const jsonPath = `${externalLocalizations.path}/${language}.json`;
                        strings = (await this.bundle.loadResource(jsonPath));
                    }
                    break;
                case "json/single-file" /* jsonSingleFile */:
                    // The bundle contains single JSON file with all strings dictionary in it.
                    strings = (await this.bundle.loadResource(externalLocalizations.path))[language];
                    break;
                case "loctable" /* loctable */:
                    throw new Error("Loctable format not supported in JS implementation");
                case "js" /* js */:
                    throw new Error("Not yet implemented");
                default:
                    throw new Error(`Unknown localization format: ${JSON.stringify(format)}`);
            }
        }
        if (environment_1.isNothing(strings)) {
            throw new Error(`Missing strings for ${language}`);
        }
        return new localized_strings_json_object_1.LocalizedStringsJSONObject(strings);
    }
}
exports.LocalizedStringsBundle = LocalizedStringsBundle;
