"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.searchTermFromRefURL = exports.extractSiriRefAppFromRefURL = exports.idTypeForMetricsOptions = exports.targetTypeForMetricsOptions = void 0;
const optional_1 = require("../../types/optional");
const urls = require("../../util/urls");
/**
 * Returns click target type for given base metrics options.
 * @param options - Base metrics options to derive click target type for.
 */
function targetTypeForMetricsOptions(options) {
    let type = options.targetType;
    if ((0, optional_1.isNothing)(type)) {
        type = "lockup" /* MetricsClickTargetType.lockup */;
    }
    return type;
}
exports.targetTypeForMetricsOptions = targetTypeForMetricsOptions;
/**
 * Returns metrics ID type for given content metrics options.
 * @param options - Content metrics options to derive metrics ID type for.
 */
function idTypeForMetricsOptions(options) {
    let type = options.idType;
    if ((0, optional_1.isNothing)(type)) {
        type = "its_id" /* MetricsIDType.itsID */;
    }
    return type;
}
exports.idTypeForMetricsOptions = idTypeForMetricsOptions;
/**
 * Extract and return Siri reference app from URL string.
 * @param refUrlString - URL string.
 * @returns An optional Siri reference app string.
 */
function extractSiriRefAppFromRefURL(urlString) {
    const refUrl = new urls.URL(urlString);
    if ((0, optional_1.isNothing)(refUrl.query)) {
        return null;
    }
    let extractedRefApp = null;
    for (const key of Object.keys(refUrl.query)) {
        if (key === "referrer") {
            if (refUrl.query[key] === "siri") {
                extractedRefApp = "com.apple.siri";
            }
            break;
        }
    }
    return extractedRefApp;
}
exports.extractSiriRefAppFromRefURL = extractSiriRefAppFromRefURL;
/**
 * Extract and return search term from reference URL string.
 * @param refUrlString - Reference URL string.
 * @returns An optional search term string.
 */
function searchTermFromRefURL(refUrlString) {
    const refUrl = new urls.URL(refUrlString);
    const queryItems = refUrl.query;
    if ((0, optional_1.isNothing)(queryItems)) {
        return null;
    }
    const searchTerm = queryItems["term"];
    const path = refUrl.pathname;
    if ((0, optional_1.isNothing)(searchTerm) || (0, optional_1.isNothing)(path)) {
        return null;
    }
    if (!path.endsWith("/search")) {
        return null;
    }
    // the url object has already url-decoded this query parameter
    const plainTerm = searchTerm;
    return plainTerm;
}
exports.searchTermFromRefURL = searchTermFromRefURL;
//# sourceMappingURL=util.js.map