//
//  date.ts
//  AppStoreKit
//
//  Created by Sam Vafaee on 10/25/17.
//  Copyright (c) 2017 Apple Inc. All rights reserved.
//
import * as serverData from "../json-parsing/server-data";
/**
 * Returns the date representation for a string ignoring the time. For preorders, it
 * is vital that any date string be passed through this before being used.
 * @param dateString  The date string to parse
 * @returns A date in local time zone.
 */
export function parseDateOmittingTimeFromString(dateString) {
    // Sanity check
    if (serverData.isNull(dateString) || dateString === "") {
        return null;
    }
    // Only consider date
    if (dateString.indexOf("T") !== -1) {
        dateString = dateString.split("T")[0];
    }
    // This string replacement is needed to ensure midnight in the local time zone
    // is used, rather than UTC-midnight.
    return new Date(dateString.replace(/-/g, "/"));
}
/**
 * Strips time from the given date and returns it as midnight UTC of the same day.
 * @param date  Local date.
 * @returns UTC date
 */
export function convertLocalDateToUTCMidnight(date) {
    // Sanity check
    if (serverData.isNull(date)) {
        return null;
    }
    return new Date(Date.UTC(date.getFullYear(), date.getMonth(), date.getDate()));
}
/**
 * Returns the number of milliseconds from epoch to UTC midnight of the given date.
 * @param date  Local date.
 * @returns Milliseconds from epoch.
 */
export function millisecondsToUTCMidnightFromLocalDate(date) {
    // Sanity check
    if (serverData.isNull(date)) {
        return null;
    }
    const utcDate = convertLocalDateToUTCMidnight(date);
    if (serverData.isNull(utcDate)) {
        return null;
    }
    return utcDate.getTime();
}
/**
 * Removes the time portion from the given date and returns it as midnight of the same day.
 * @param date  Local date
 * @returns Local date
 */
export function convertLocalDateToLocalMidnight(date) {
    // Sanity check
    if (serverData.isNull(date)) {
        return null;
    }
    const midnight = new Date(date);
    midnight.setHours(0, 0, 0, 0);
    return midnight;
}
/**
 * Sets the minutes, seconds and milliseconds of the given date to 0.
 * @param date  Local date
 * @returns Local date
 */
export function convertLocalDateByFlooringToHour(date) {
    // Sanity check
    if (serverData.isNull(date)) {
        return null;
    }
    return dateFlooredToHour(date);
}
/**
 * Creates a new date with the minutes, seconds and milliseconds of the given date set to 0.
 * @param date The date to remove minutes, seconds, and milliseconds from.
 * @returns The new date with minutes, seconds, and milliseconds removed using local device time zone.
 */
export function dateFlooredToHour(date) {
    const dateCopy = new Date(date.getTime());
    dateCopy.setMinutes(0);
    dateCopy.setSeconds(0);
    dateCopy.setMilliseconds(0);
    return dateCopy;
}
/**
 * Returns the number of hours between two dates, or null if either date is null.
 * @param fromDate The date to calculate from.
 * @param toDate The date to calculate to.
 */
export function numberOfHoursBetween(fromDate, toDate) {
    // Sanity check
    if (serverData.isNull(fromDate) || serverData.isNull(toDate)) {
        return null;
    }
    return Math.ceil((toDate.getTime() - dateFlooredToHour(fromDate).getTime()) / (60 * 60 * 1000));
}
/**
 * Returns the number of days between two dates, or null if either date is null.
 * @param fromDate The date to calculate from.
 * @param toDate The date to calculate to.
 */
export function numberOfDaysBetween(fromDate, toDate) {
    // Sanity check
    if (serverData.isNull(fromDate) || serverData.isNull(toDate)) {
        return null;
    }
    return Math.floor((toDate.getTime() - fromDate.getTime()) / (60 * 60 * 1000 * 24));
}
/**
 * Whether or not two dates occur on the same day.
 * @param date A date that will be compared to the second input.
 * @param otherDate A date that will be compared to the first input.
 */
export function areLocalDatesSameDay(date, otherDate) {
    var _a, _b;
    return ((_a = convertLocalDateToLocalMidnight(date)) === null || _a === void 0 ? void 0 : _a.getTime()) === ((_b = convertLocalDateToLocalMidnight(otherDate)) === null || _b === void 0 ? void 0 : _b.getTime());
}
//# sourceMappingURL=date-util.js.map