import { isNothing } from "@jet/environment/types/optional";
import * as serverData from "../json-parsing/server-data";
import * as attributes from "./attributes";
/**
 * There are nested attributes that are platform-specific. If provided with a platform for which the response data has
 * platform-specific attributes, this function will return those attributes.
 * @param {Data} data The data for which to determine attributes.
 * @param {AttributePlatform} platform The platform to fetch the attributes for
 * @returns {any} If a platform is provided, returns `true` exactly when platform-specific attributes exist for that
 * platform. If no platform is provided, it simply returns the data's top-level attributes.
 */
function attributesForPlatform(data, platform) {
    const allPlatformAttributes = attributes.attributeAsDictionary(data, "platformAttributes");
    return serverData.traverse(allPlatformAttributes, platform !== null && platform !== void 0 ? platform : undefined);
}
/**
 * Determines if attributes exist for a given platform
 * @param {Data} data The data to check
 * @param {AttributePlatform} platform The platform to check
 * @returns {boolean} True if the platform exists in the data's platform attributes. False if not.
 */
export function hasPlatformAttribute(data, platform) {
    const platformAttributes = attributesForPlatform(data, platform);
    return serverData.isDefinedNonNullNonEmpty(platformAttributes);
}
/**
 * Retrieve the specified attribute from the data as a dictionary
 *
 * @param data The data from which to retrieve the attribute.
 * @param platform The platform to look up
 * @param attributePath The path of the attribute.
 * @returns The value for the requested attribute
 */
export function platformAttributeAsDictionary(data, platform, attributePath) {
    const platformAttributes = attributesForPlatform(data, platform);
    if (serverData.isNull(platformAttributes)) {
        return null;
    }
    return serverData.asDictionary(platformAttributes, attributePath);
}
/**
 * Retrieve the specified attribute from the data as an array.
 *
 * @param data The data from which to retrieve the attribute.
 * @param platform The platform to look up
 * @param attributePath The path of the attribute.
 * @returns {any[]} The attribute value as an array.
 */
export function platformAttributeAsArray(data, platform, attributePath) {
    const platformAttributes = attributesForPlatform(data, platform);
    if (isNothing(platformAttributes)) {
        return null;
    }
    return serverData.asArray(platformAttributes, attributePath);
}
/**
 * Retrieve the specified attribute from the data as an array, coercing to an empty array if the object is not an array.
 *
 * @param data The data from which to retrieve the attribute.
 * @param platform The platform to look up
 * @param attributePath The path of the attribute.
 * @returns {any[]} The attribute value as an array.
 */
export function platformAttributeAsArrayOrEmpty(data, platform, attributePath) {
    const platformAttributes = attributesForPlatform(data, platform);
    if (serverData.isNull(platformAttributes)) {
        return [];
    }
    return serverData.asArrayOrEmpty(platformAttributes, attributePath);
}
/**
 * Retrieve the specified attribute from the data as a string.
 *
 * If the attribute lives under the platform-specific attributes, then a platform may be provided to properly call in to
 * the nested structure.
 * @param data The data from which to retrieve the attribute.
 * @param platform The platform to look up
 * @param attributePath The object path for the attribute.
 * @param policy The validation policy to use when resolving this value.
 * @returns {string} The attribute value as a string.
 */
export function platformAttributeAsString(data, platform, attributePath, policy = "coercible") {
    const platformAttributes = attributesForPlatform(data, platform);
    if (serverData.isNull(platformAttributes)) {
        return null;
    }
    return serverData.asString(platformAttributes, attributePath, policy);
}
/**
 * Retrieve the specified attribute from the data as a boolean.
 *
 * If the attribute lives under the platform-specific attributes, then a platform may be provided to properly call in to
 * the nested structure.
 * @param data The data from which to retrieve the attribute.
 * @param platform The platform to look up
 * @param attributePath The path of the attribute.
 * @param policy The validation policy to use when resolving this value.
 * @returns {boolean} The attribute value as a boolean.
 */
export function platformAttributeAsBoolean(data, platform, attributePath, policy = "coercible") {
    const platformAttributes = attributesForPlatform(data, platform);
    if (serverData.isNull(platformAttributes)) {
        return null;
    }
    return serverData.asBoolean(platformAttributes, attributePath, policy);
}
/**
 * Retrieve the specified attribute from the data as a boolean, which will be `false` if the attribute does not exist.
 *
 * If the attribute lives under the platform-specific attributes, then a platform may be provided to properly call in to
 * the nested structure.
 * @param data The data from which to retrieve the attribute.
 * @param platform The platform to look up
 * @param attributePath The path of the attribute.
 * @returns {boolean} The attribute value as a boolean, coercing to `false` if the value is not present..
 */
export function platformAttributeAsBooleanOrFalse(data, platform, attributePath) {
    const platformAttributes = attributesForPlatform(data, platform);
    if (serverData.isNull(platformAttributes)) {
        return false;
    }
    return serverData.asBooleanOrFalse(platformAttributes, attributePath);
}
/**
 * Retrieve the specified attribute from the data as a number.
 *
 * If the attribute lives under the platform-specific attributes, then a platform may be provided to properly call in to
 * the nested structure.
 * @param data The data from which to retrieve the attribute.
 * @param platform The platform to look up
 * @param attributePath The path of the attribute.
 * @param policy The validation policy to use when resolving this value.
 * @returns {boolean} The attribute value as a number.
 */
export function platformAttributeAsNumber(data, platform, attributePath, policy = "coercible") {
    const platformAttributes = attributesForPlatform(data, platform);
    if (serverData.isNull(platformAttributes)) {
        return null;
    }
    return serverData.asNumber(platformAttributes, attributePath, policy);
}
// endregion
//# sourceMappingURL=platform-attributes.js.map