import { isNothing } from "@jet/environment/types/optional";
import * as serverData from "../json-parsing/server-data";
// region Generic Attribute retrieval
// region Attribute retrieval
/**
 * Retrieve the specified attribute from the data, coercing it to a JSONData dictionary
 *
 * @param data The data from which to retrieve the attribute.
 * @param attributePath The path of the attribute.
 * @param defaultValue The object to return if the path search fails.
 * @returns The dictionary of data
 */
export function attributeAsDictionary(data, attributePath, defaultValue) {
    if (isNothing(data)) {
        return null;
    }
    return serverData.asDictionary(data.attributes, attributePath, defaultValue);
}
/**
 * Retrieve the specified attribute from the data, coercing it to an Interface
 *
 * @param data The data from which to retrieve the attribute.
 * @param attributePath The path of the attribute.
 * @param defaultValue The object to return if the path search fails.
 * @returns The dictionary of data as an interface
 */
export function attributeAsInterface(data, attributePath, defaultValue) {
    return attributeAsDictionary(data, attributePath, defaultValue);
}
/**
 * Retrieve the specified attribute from the data as an array, coercing to a JSONValue array
 *
 * @param data The data from which to retrieve the attribute.
 * @param attributePath The path of the attribute.
 * @returns {any[]} The attribute value as an array.
 */
export function attributeAsArray(data, attributePath) {
    if (serverData.isNull(data)) {
        return [];
    }
    return serverData.asArray(data.attributes, attributePath);
}
/**
 * Retrieve the specified attribute from the data as an array, coercing to an empty array if the object is not an array.
 *
 * @param data The data from which to retrieve the attribute.
 * @param attributePath The path of the attribute.
 * @returns {any[]} The attribute value as an array.
 */
export function attributeAsArrayOrEmpty(data, attributePath) {
    var _a;
    return (_a = attributeAsArray(data, attributePath)) !== null && _a !== void 0 ? _a : [];
}
/**
 * Retrieve the specified attribute from the data as a string.
 *
 * @param data The data from which to retrieve the attribute.
 * @param attributePath The object path for the attribute.
 * @param policy The validation policy to use when resolving this value.
 * @returns {string} The attribute value as a string.
 */
export function attributeAsString(data, attributePath, policy = "coercible") {
    if (serverData.isNull(data)) {
        return null;
    }
    return serverData.asString(data.attributes, attributePath, policy);
}
/**
 * Retrieve the specified attribute from the data as a boolean.
 *
 * @param data The data from which to retrieve the attribute.
 * @param attributePath The path of the attribute.
 * @param policy The validation policy to use when resolving this value.
 * @returns {boolean} The attribute value as a boolean.
 */
export function attributeAsBoolean(data, attributePath, policy = "coercible") {
    if (serverData.isNull(data)) {
        return null;
    }
    return serverData.asBoolean(data.attributes, attributePath, policy);
}
/**
 * Retrieve the specified attribute from the data as a boolean, which will be `false` if the attribute does not exist.
 *
 * @param data The data from which to retrieve the attribute.
 * @param attributePath The path of the attribute.
 * @returns {boolean} The attribute value as a boolean, coercing to `false` if the value is not present..
 */
export function attributeAsBooleanOrFalse(data, attributePath) {
    if (serverData.isNull(data)) {
        return false;
    }
    return serverData.asBooleanOrFalse(data.attributes, attributePath);
}
/**
 * Retrieve the specified attribute from the data as a number.
 *
 * @param data The data from which to retrieve the attribute.
 * @param attributePath The path of the attribute.
 * @param policy The validation policy to use when resolving this value.
 * @returns {boolean} The attribute value as a number.
 */
export function attributeAsNumber(data, attributePath, policy = "coercible") {
    if (serverData.isNull(data)) {
        return null;
    }
    return serverData.asNumber(data.attributes, attributePath, policy);
}
export function hasAttributes(data) {
    return !serverData.isNull(serverData.asDictionary(data, "attributes"));
}
/**
 * The canonical way to detect if an item from Media API is hydrated or not.
 *
 * @param data The data from which to retrieve the attributes.
 */
export function isNotHydrated(data) {
    return !hasAttributes(data);
}
// region Custom Attributes
/**
 * Performs conversion for a custom variant of given attribute, if any are available.
 * @param attribute Attribute to get custom attribute key for, if any.
 */
export function attributeKeyAsCustomAttributeKey(attribute) {
    return customAttributeMapping[attribute];
}
/**
 * Whether or not given custom attributes key allows fallback to default page with AB testing treatment within a nondefault page.
 * This is to allow AB testing to affect only icons within custom product pages.
 */
export function attributeAllowsNonDefaultTreatmentInNonDefaultPage(customAttribute) {
    return customAttribute === "customArtwork" || customAttribute === "customIconArtwork"; // Only the icon artwork.
}
/**
 * Defines mapping of attribute to custom attribute.
 */
const customAttributeMapping = {
    artwork: "customArtwork",
    iconArtwork: "customIconArtwork",
    screenshotsByType: "customScreenshotsByType",
    promotionalText: "customPromotionalText",
    videoPreviewsByType: "customVideoPreviewsByType",
    customScreenshotsByTypeForAd: "customScreenshotsByTypeForAd",
    customVideoPreviewsByTypeForAd: "customVideoPreviewsByTypeForAd",
    customDeepLink: "customDeepLink",
};
// endregion
//# sourceMappingURL=attributes.js.map