import { isSome } from "@jet/environment/types/optional";
import { makeMetatype } from "@jet/environment/util/metatype";
import { unreachable } from "../util/errors";
export const ActiveIntentMetaType = makeMetatype("app-store:active-intent");
/**
 * Executes {@linkcode callback} with an {@linkcode AppStoreObjectGraph} that has been prepared with an
 * {@linkcode ActiveIntent} dependency
 *
 * This will allow code running within {@linkcode callback} to access meta-data about the active
 * {@linkcode Intent} without needing that information to be passed through every layer of the code
 */
export function withActiveIntent(objectGraph, intent, callback) {
    const objectGraphWithActiveIntent = objectGraph.addingActiveIntent({
        previewPlatform: intent.platform,
    });
    return callback(objectGraphWithActiveIntent);
}
/**
 * Ephemeral storage of `Intent`-specific details that need to be accessible
 * throughout the process of building that `Intent`'s result
 */
export class ActiveIntent {
    constructor(implementation) {
        this.implementation = implementation;
        this.inferredPreviewPlatform = undefined;
    }
    /**
     * Explicitly set a {@linkcode PreviewPlatform} for the active intent
     *
     * This might be a value derrived from an API response, in cases where the original `Intent`
     * was not specific about which platform to display
     */
    setInferredPreviewPlatform(platform) {
        this.inferredPreviewPlatform = platform;
    }
    /**
     * The {@linkcode PreviewPlatform} value of the current {@linkcode Intent}
     */
    get previewPlatform() {
        var _a;
        return (_a = this.inferredPreviewPlatform) !== null && _a !== void 0 ? _a : this.implementation.previewPlatform;
    }
    /**
     * The {@linkcode Platform} equivalent of the active {@linkcode PreviewPlatform}
     *
     * This translates the user-facing {@linkcode PreviewPlatform} into the "internal"
     * Media API {@linkcode Platform}
     */
    get platform() {
        if (isSome(this.previewPlatform)) {
            switch (this.previewPlatform) {
                case "ipad":
                case "iphone":
                case "mac":
                case "watch":
                    return this.previewPlatform;
                case "tv":
                    return "appletv";
                case "vision":
                    return "realityDevice";
                default:
                    unreachable(this.previewPlatform);
            }
        }
        return undefined;
    }
    /**
     * The {@linkcode AttributePlatform} equivalent of the active {@linkcode PreviewPlatform}
     *
     * This translates the user-facing {@linkcode PreviewPlatform} into the "internal"
     * {@linkcode AttributePlatform}
     */
    get attributePlatform() {
        if (isSome(this.previewPlatform)) {
            switch (this.previewPlatform) {
                case "iphone":
                case "ipad":
                    return "ios";
                case "mac":
                    return "osx";
                case "tv":
                    return "appletvos";
                case "vision":
                    return "xros";
                case "watch":
                    return "watch";
                default:
                    unreachable(this.previewPlatform);
            }
        }
        return undefined;
    }
    /**
     * The {@linkcode AppPlatform} equivalent of the active {@linkcode PreviewPlatform}
     *
     * This translates the user-facing {@linkcode PreviewPlatform} into the "internal"
     * {@linkcode AppPlatform}
     */
    get appPlatform() {
        if (isSome(this.previewPlatform)) {
            switch (this.previewPlatform) {
                case "ipad":
                    return "pad";
                case "iphone":
                    return "phone";
                case "mac":
                    return "mac";
                case "tv":
                    return "tv";
                case "vision":
                    return "vision";
                case "watch":
                    return "watch";
                default:
                    unreachable(this.previewPlatform);
            }
        }
        return undefined;
    }
}
//# sourceMappingURL=active-intent.js.map