import { withActiveIntent } from "../../foundation/dependencies/active-intent";
import { FlowAction } from "../../api/models";
import { productPageRoutes, makeProductPageURLWithoutAppSlug, } from "../../common/product-page/intent-controller-routing";
import { injectSEOData } from "../../api/models/web-renderable-page";
import { lookupURLForProductId as makeProductPageRequest } from "../../common/builders/url-mapping";
import { createProductPageFromResponse } from "../../common/product-page/product-page-util";
import { injectWebNavigation } from "../../common/web-navigation/inject-web-navigation";
import * as mediaNetwork from "../../foundation/media/network";
import { URL } from "../../foundation/network/urls";
import { inferPreviewPlatformFromDeviceFamilies } from "../../common/preview-platform";
import { validateAdamId } from "../../foundation/media/util";
export const ProductPageIntentController = {
    $intentKind: "ProductPageIntent",
    routes: productPageRoutes,
    async perform(intent, objectGraphWithoutActiveIntent) {
        return await withActiveIntent(objectGraphWithoutActiveIntent, intent, async (objectGraph) => {
            var _a, _b;
            // See: https://github.pie.apple.com/its/Jingle/blob/ce14e21b6ac3dd4be884aa12b26d4e79c6d8aa7a/MZStorePlatform/src/main/java/com/apple/jingle/store/mediaapi/resource/SFMediaAPICommonResourceType.java#L46
            validateAdamId(objectGraph, intent.id);
            const mediaApiRequest = makeProductPageRequest(objectGraph, intent.id, false, intent.ppid, false, false);
            const response = await mediaNetwork.fetchData(objectGraph, mediaApiRequest);
            // If the `Intent` does not specify a platform that the user requested, we need to select one of the options
            // available in the Media API response to use instead. This will ensure we use the correct platform attributes
            // when rendering the view-model
            const inferredPlatform = inferPreviewPlatformFromDeviceFamilies(objectGraph, response);
            if (!objectGraph.activeIntent.previewPlatform) {
                const platformToSet = (_a = intent.platform) !== null && _a !== void 0 ? _a : inferredPlatform;
                objectGraph.activeIntent.setInferredPreviewPlatform(platformToSet);
            }
            const previewPlatform = objectGraph.activeIntent.previewPlatform;
            const productPage = (await createProductPageFromResponse(objectGraph, response));
            // The page URL (`.canonicalURL` property on the view-model) from Media API does not account for
            // the `platform` and `ppid` query param; we need to add that in ourselves.
            // We can't rely on `makeCanonicalUrl` as that does not handle the `appName` segment;
            // We need to take the Media API URL and append the `platform` and `ppid` ourselves
            if (productPage.canonicalURL) {
                const pageURL = new URL(productPage.canonicalURL);
                if (typeof intent.platform === "string" && intent.platform !== inferredPlatform) {
                    pageURL.param("platform", intent.platform);
                }
                if (typeof intent.ppid === "string" && intent.ppid) {
                    pageURL.param("ppid", intent.ppid);
                }
                if (typeof intent.lic === "string" && intent.lic) {
                    pageURL.param("lic", intent.lic);
                }
                productPage.canonicalURL = pageURL.toString();
            }
            if (objectGraph.client.isWeb) {
                injectWebNavigation(objectGraph, productPage, previewPlatform);
                injectSEOData(productPage, (_b = objectGraph.seo) === null || _b === void 0 ? void 0 : _b.getSEODataForProductPage(objectGraph, productPage, response));
            }
            return productPage;
        });
    },
    actionFor(intent, objectGraph) {
        const action = new FlowAction("product");
        action.destination = intent;
        action.pageUrl = makeProductPageURLWithoutAppSlug(objectGraph, intent);
        return action;
    },
};
//# sourceMappingURL=product-page-intent-controller.js.map