import { isNothing } from "@jet/environment";
import { MetricsIdentifierType } from "../../foundation/metrics/metrics-identifiers-cache";
export function isFeatureEnabledForCurrentUser(objectGraph, rolloutRate) {
    var _a;
    if (rolloutRate <= 0.0) {
        return false;
    }
    if (rolloutRate >= 1.0) {
        // It is important that we not require a metrics ID when our rollout rate is 100%
        return true;
    }
    const metricsIdString = (_a = objectGraph.metricsIdentifiersCache) === null || _a === void 0 ? void 0 : _a.getMetricsIdForType(MetricsIdentifierType.user);
    if (isNothing(metricsIdString) || metricsIdString.length === 0) {
        // if the user does not have a DSID, err on the side of caution.
        return false;
    }
    if (metricsIdString.length < 2) {
        // If we don't have enough characters to normalize, which should never happen, safely return false.
        return false;
    }
    return normalizedMetricsIdLotteryValue(metricsIdString) < rolloutRate;
}
/**
 * Gets a normalized value based on the metrics ID to match against the rollout rate to bucket a user.
 * @param metricsId The user's metrics ID string
 * @returns A value from 0.0 to 1.0 that we use to bucket the user
 */
function normalizedMetricsIdLotteryValue(metricsId) {
    // Character mapping function that converts a character to its corresponding number.
    // This value is guaranteed to be between 0 and 61, inclusive.
    function charToNumber(char) {
        const charCode = char.charCodeAt(0);
        if (charCode >= 48 && charCode <= 57) {
            // '0'-'9'
            return 52 + charCode - 48;
        }
        else if (charCode >= 65 && charCode <= 90) {
            // 'A'-'Z'
            return 26 + charCode - 65;
        }
        else if (charCode >= 97 && charCode <= 122) {
            // 'a'-'z'
            return charCode - 97;
        }
        // This should never happen, if it does, the metrics ID is corrupt, but we rely on that not happening.
        // This would indeed break the value guarantee.
        return charCode;
    }
    // Extract the last two characters from the string.
    const lastTwoChars = metricsId.slice(-2);
    // Convert each of the last two characters to their respective number representation.
    const num1 = charToNumber(lastTwoChars[0]);
    const num2 = charToNumber(lastTwoChars[1]);
    // Combine the two numbers into a single base-62 number.
    const combinedValue = num1 * 62 + num2;
    // Calculate the maximum possible value for two characters (each char has a value of 0 to 61, inclusive).
    const maxValue = 61 * 62 + 61;
    // Normalize the combined value to be between 0.0 and 1.0.
    const normalizedValue = combinedValue / maxValue;
    return normalizedValue;
}
//# sourceMappingURL=lottery.js.map