import * as validation from "@jet/environment/json/validation";
import { TodayCardMediaRiver } from "../../../api/models";
import { isDarkColor, rgbWith } from "../../../foundation/util/color-util";
import { applyTodayCardOverridesForAcquisitionStoryIfNecessary, relatedContentOverlayFromData, } from "../today-card-overlay-util";
import { applyMultiAppFallbackToCollectionCard, cardDisplayStyleFromData, gridFallbackLimit, lockupsForCollectionCardFromData, popTodayCardLocation, pushTodayCardLocation, } from "../today-card-util";
import { createTodayBaseCard } from "./today-base-card-builder";
/**
 * Create TodayCard displaying the river of lockups
 *
 * @param objectGraph The dependency graph for the App Store
 * @param data The media api data to build the card from
 * @param cardConfig The configuration for the card
 * @param context The parse context for the over all today page
 * @param augmentingData that stores some additional responses that may be used to enhance the contents of `data`
 * @returns The newly created TodayCard, using a river media.
 */
export function createTodayRiverCard(objectGraph, data, cardConfig, context, augmentingData) {
    return validation.context("createTodayRiverCard", () => {
        const riverCard = createTodayBaseCard(objectGraph, data, cardConfig, context);
        cardConfig.useJoeColorIconPlaceholder = true;
        pushTodayCardLocation(objectGraph, data, cardConfig, context);
        const shouldIncludeLockupClickActions = objectGraph.client.isWeb;
        const riverLockups = lockupsForCollectionCardFromData(objectGraph, data, cardConfig, context, shouldIncludeLockupClickActions);
        if (riverLockups.length === 0) {
            popTodayCardLocation(context);
            return null;
        }
        cardConfig.canDisplayArcadeOfferButton = false;
        if (riverLockups.length < gridFallbackLimit(objectGraph)) {
            // <rdar://problem/37261537> P2: Some Today cards not showing on Cinar
            // less that `gridFallbackLimit` lockups gets a fallback card on Emet only
            applyMultiAppFallbackToCollectionCard(objectGraph, data, riverLockups, riverCard);
        }
        else {
            riverCard.style = "dark";
            replaceHighLuminanceIconColorsInLockups(riverLockups);
            riverCard.media = new TodayCardMediaRiver(riverLockups);
        }
        if (objectGraph.client.isWatch) {
            riverCard.overlay = relatedContentOverlayFromData(objectGraph, riverCard, cardConfig, data, riverLockups);
        }
        // Special post-processing step for Acquisition story cards.
        // This is needed to splice in data not included in initial response.
        applyTodayCardOverridesForAcquisitionStoryIfNecessary(objectGraph, riverCard, cardConfig, cardDisplayStyleFromData(data, cardConfig.coercedCollectionTodayCardDisplayStyle), data, augmentingData, context);
        popTodayCardLocation(context);
        return riverCard;
    });
}
/**
 * Replace any icon background colors with luminance > 0.9 with grey color.
 * This ensure that river background colors don't compete with the overlayed white text.
 *
 * @param riverLockups The river lockups.
 */
function replaceHighLuminanceIconColorsInLockups(riverLockups) {
    for (const lockup of riverLockups) {
        if (!isDarkColor(lockup.icon.backgroundColor, 90)) {
            lockup.icon.backgroundColor = rgbWith(0.8, 0.8, 0.8);
        }
    }
}
//# sourceMappingURL=today-river-card-builder.js.map