import * as validation from "@jet/environment/json/validation";
import { TodayCardInAppPurchase } from "../../../api/models";
import { isNullOrEmpty } from "../../../foundation/json-parsing/server-data";
import { inAppPurchaseLockupFromData } from "../../lockups/lockups";
import { applyTodayCardOverridesForAcquisitionStoryIfNecessary } from "../today-card-overlay-util";
import { cardDisplayStyleFromData, popTodayCardLocation, pushTodayCardLocation, relatedCardContentsContentsFromData, } from "../today-card-util";
import { createTodayBaseCard } from "./today-base-card-builder";
/**
 * Create TodayCard displaying an in app purchase
 *
 * @param objectGraph The dependency graph for the App Store
 * @param data The media api data to build the card from
 * @param cardConfig The configuration for the card
 * @param context The parse context for the over all today page
 * @param augmentingData that stores some additional responses that may be used to enhance the contents of `data`
 * @returns The newly created TodayCard, using the in app purchase media.
 */
export function createTodayInAppPurchaseCard(objectGraph, data, cardConfig, context, augmentingData) {
    return validation.context("createTodayInAppPurchaseCard", () => {
        const inAppPurchaseCard = createTodayBaseCard(objectGraph, data, cardConfig, context);
        pushTodayCardLocation(objectGraph, data, cardConfig, context);
        const iAPData = inAppPurchaseDataFromRelatedContent(objectGraph, relatedCardContentsContentsFromData(objectGraph, data));
        if (isNullOrEmpty(iAPData)) {
            popTodayCardLocation(context);
            return null;
        }
        const lockup = inAppPurchaseLockupFromData(objectGraph, iAPData, {
            offerStyle: "colored",
            metricsOptions: {
                pageInformation: context.pageInformation,
                locationTracker: context.locationTracker,
            },
            artworkUseCase: 1 /* ArtworkUseCase.LockupIconSmall */,
        });
        if (isNullOrEmpty(lockup)) {
            popTodayCardLocation(context);
            return null;
        }
        lockup.theme = "infer";
        inAppPurchaseCard.media = new TodayCardInAppPurchase(lockup);
        // we need to add impressions to the media here because
        // there is no overlay on an IAP inAppPurchaseCard for some reason
        inAppPurchaseCard.media.impressionMetrics = lockup.impressionMetrics;
        inAppPurchaseCard.media.impressionMetrics.fields["parentId"] = inAppPurchaseCard.impressionMetrics.fields["id"];
        inAppPurchaseCard.style = "white";
        // Special post-processing step for Acquisition story cards.
        // This is needed to splice in data not included in initial response.
        applyTodayCardOverridesForAcquisitionStoryIfNecessary(objectGraph, inAppPurchaseCard, cardConfig, cardDisplayStyleFromData(data, cardConfig.coercedCollectionTodayCardDisplayStyle), data, augmentingData, context);
        popTodayCardLocation(context);
        return inAppPurchaseCard;
    });
}
function inAppPurchaseDataFromRelatedContent(objectGraph, relatedContent) {
    if (relatedContent.length === 1) {
        const contentData = relatedContent[0];
        if (contentData.type === "in-apps") {
            return contentData;
        }
    }
    return null;
}
//# sourceMappingURL=today-in-app-purchase-card-builder.js.map