import { isSome } from "@jet/environment";
import * as validation from "@jet/environment/json/validation";
import { isNothing } from "@jet/environment/types/optional";
import { TodayCardLockupOverlay, TodayCardMediaAppEvent, } from "../../../api/models";
import { asBooleanOrFalse } from "../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as mediaRelationship from "../../../foundation/media/relationships";
import * as color from "../../../foundation/util/color-util";
import { appEventOrPromotionStartDateFromData } from "../../app-promotions/app-event";
import { appEventsAreEnabled } from "../../app-promotions/app-promotions-common";
import { addNextPreferredContentRefreshDate } from "../../refresh/page-refresh-controller";
import { todayCardArtworkDetails } from "../artwork/today-artwork-util";
import { applyTodayCardOverridesForAcquisitionStoryIfNecessary } from "../today-card-overlay-util";
import { cardDisplayStyleFromData, cardStyleFromJoeColorsWithoutFallback, offerEnvironmentForTodayCard, offerStyleForTodayCard, popTodayCardLocation, pushTodayCardLocation, } from "../today-card-util";
import { createTodayBaseCard } from "./today-base-card-builder";
/**
 * Create TodayCard displaying an app event
 *
 * @param objectGraph The dependency graph for the App Store
 * @param data The media api data to build the card from
 * @param cardConfig The configuration for the card
 * @param context The parse context for the over all today page
 * @param augmentingData that stores some additional responses that may be used to enhance the contents of `data`
 * @returns The newly created TodayCard, using a app event media.
 */
export function createTodayAppEventCard(objectGraph, data, cardConfig, context, augmentingData) {
    return validation.context("createTodayAppEventCard", () => {
        var _a, _b, _c, _d, _e;
        if (!appEventsAreEnabled(objectGraph)) {
            return null;
        }
        const appEventData = appEventDataFromData(objectGraph, data);
        if (isNothing(appEventData)) {
            return null;
        }
        const appEventCard = createTodayBaseCard(objectGraph, data, cardConfig, context, (clickOptions) => {
            // Add app event ID to click options
            clickOptions.inAppEventId = appEventData.id;
            const parentAppData = mediaRelationship.relationshipData(objectGraph, appEventData, "app");
            if (isSome(parentAppData)) {
                clickOptions.relatedSubjectIds = [parentAppData.id];
            }
        });
        if (isSome(appEventCard.editorialDisplayOptions)) {
            appEventCard.editorialDisplayOptions.useMaterialBlur = true;
        }
        pushTodayCardLocation(objectGraph, data, cardConfig, context);
        // Artwork
        const mediaDetails = todayCardArtworkDetails(objectGraph, data, cardConfig);
        if (isNothing(mediaDetails)) {
            popTodayCardLocation(context);
            return null;
        }
        // Tint and styling
        const tintColor = (_b = (_a = mediaDetails.joeColors) === null || _a === void 0 ? void 0 : _a.textColor4) !== null && _b !== void 0 ? _b : color.black;
        const artworkBackgroundColor = (_d = (_c = mediaDetails.joeColors) === null || _c === void 0 ? void 0 : _c.bgColor) !== null && _d !== void 0 ? _d : color.black;
        const blurStyle = color.isDarkColor(artworkBackgroundColor) ? "dark" : "light";
        // App event
        const metricsOptions = {
            pageInformation: context.pageInformation,
            locationTracker: context.locationTracker,
            targetType: "eventModule",
        };
        // Card style
        appEventCard.style =
            (_e = cardStyleFromJoeColorsWithoutFallback(mediaDetails.joeColors, "textColor4")) !== null && _e !== void 0 ? _e : (color.isDarkColor(tintColor) ? "dark" : "light");
        // Offer style
        const offerStyle = offerStyleForTodayCard(objectGraph, appEventCard.style);
        const editorialKind = mediaAttributes.attributeAsString(data, "label");
        const appEventOrDate = appEventOrPromotionStartDateFromData(objectGraph, appEventData, null, false, false, offerEnvironmentForTodayCard(appEventCard.style), offerStyle, true, metricsOptions, true, true, editorialKind, false, asBooleanOrFalse(cardConfig.allowUnpublishedAppEventPreviews));
        // Return early if we received a Date, as this means the App Event shouldn't be shown yet.
        if (isNothing(appEventOrDate) || appEventOrDate instanceof Date) {
            addNextPreferredContentRefreshDate(appEventOrDate, context.refreshController);
            popTodayCardLocation(context);
            return null;
        }
        const appEvent = appEventOrDate;
        // Card
        appEventCard.media = new TodayCardMediaAppEvent(appEvent.formattedDates, appEvent.startDate, tintColor, mediaDetails.artworks, mediaDetails.videos, mediaDetails.artworkLayoutsWithMetrics, blurStyle);
        if (isSome(appEvent.lockup)) {
            appEventCard.overlay = new TodayCardLockupOverlay(appEvent.lockup);
        }
        // Special post-processing step for Acquisition story cards.
        // This is needed to splice in data not included in initial response.
        applyTodayCardOverridesForAcquisitionStoryIfNecessary(objectGraph, appEventCard, cardConfig, cardDisplayStyleFromData(data, cardConfig.coercedCollectionTodayCardDisplayStyle), data, augmentingData, context);
        popTodayCardLocation(context);
        return appEventCard;
    });
}
/**
 * @param objectGraph The dependency graph for the App Store
 * @param data The media api data to build the card from
 * @returns The app event relationship data for this card
 */
function appEventDataFromData(objectGraph, data) {
    let appEventData;
    // Primary content
    const primaryContent = mediaRelationship.relationshipCollection(data, "primary-content");
    if (primaryContent.length > 0) {
        appEventData = primaryContent[0];
    }
    else {
        // Card contents
        const cardContents = mediaRelationship.relationshipCollection(data, "card-contents");
        if (cardContents.length === 0) {
            return null;
        }
        appEventData = cardContents[0];
    }
    return appEventData;
}
//# sourceMappingURL=today-app-event-card-builder.js.map