import { isNullOrEmpty } from "../../foundation/json-parsing/server-data";
import { currentPosition } from "../metrics/helpers/location";
/**
 * Opaque token to use for paginating a list of search results.
 * This is used for both standard and segmented search results.
 */
export class SearchToken {
}
/// The number of results to load per page.
const suggestedMaxResutsPerPage = 30;
/**
 * Create a search token for loading more search results.
 * @param results Remaining set of data that is yet to be paginated
 * @param requestMetadata The nature of request that was fired.
 * @param responseMetadata The meta blob returned as part of initial search. This is preserved, as subsequent pagination requests don't hit search endpoints.
 * @param metricsOptions Metrics options to preserve during pagination
 */
export function createSearchToken(objectGraph, results, requestMetadata, responseMetadata, metricsOptions) {
    if (isNullOrEmpty(results)) {
        return null;
    }
    return {
        results: results,
        maxPerPage: suggestedMaxResutsPerPage,
        requestMetadata: requestMetadata,
        metricsOptions: metricsOptions,
        responseMetadata: responseMetadata !== null && responseMetadata !== void 0 ? responseMetadata : {},
        contentOffsetWithinResultsShelf: currentPosition(metricsOptions.locationTracker),
    };
}
/**
 * Returns the next set of items to load.
 * @param searchToken Search token used for paginating.
 */
export function getNextItemsToFetch(objectGraph, searchToken) {
    if (!searchToken || !searchToken.results) {
        return [];
    }
    return searchToken.results.slice(0, searchToken.maxPerPage);
}
/**
 * Advance the search token by the items that were loaded, consistent with `getNextItemsToFetch`
 * @param searchToken Search token to create new token with.
 */
export function advanceSearchTokenResults(objectGraph, searchToken) {
    let nextPageResults = [];
    if (searchToken && searchToken.results) {
        nextPageResults = searchToken.results.slice(searchToken.maxPerPage, searchToken.results.length);
    }
    if (isNullOrEmpty(nextPageResults)) {
        return null;
    }
    const nextToken = { ...searchToken };
    nextToken.results = nextPageResults;
    return nextToken;
}
//# sourceMappingURL=search-token.js.map