/**
 * Build methods for Search Facets.
 */
import * as models from "../../api/models";
import * as serverData from "../../foundation/json-parsing/server-data";
import { categoryListFromApiResponse } from "../categories";
import * as metricsClickHelpers from "../metrics/helpers/clicks";
/**
 * Create Search Facets. Theres platform specific variations here.
 * @param requestFacets Facets in current request.
 * @param categoryFacetsData Additional
 */
export function createSearchFacets(objectGraph, requestFacets, categoryFacetsData) {
    const selectedFacets = requestFacets || {};
    const facets = [];
    // Device Type
    if (objectGraph.client.deviceType !== "mac") {
        facets.push(new models.SearchFacetSet("targetPlatform", [
            new models.SearchFacetValue(objectGraph.loc.string("Search.Facets.iPadAndIPhone"), null, selectedFacets["targetPlatform"]),
            new models.SearchFacetValue(objectGraph.loc.string("Search.Facets.iPhoneOnly"), "iphone", selectedFacets["targetPlatform"]),
        ]));
    }
    // Price
    facets.push(new models.SearchFacetSet("price", [
        new models.SearchFacetValue(objectGraph.loc.string("SEARCH_FACET_ANY_PRICE", "Any"), null, selectedFacets["price"]),
        new models.SearchFacetValue(objectGraph.loc.string("SEARCH_FACET_FREE", "Any"), "free", selectedFacets["price"]),
    ]));
    // Categories
    const categoryList = categoryListFromApiResponse(objectGraph, categoryFacetsData, false);
    if (categoryList) {
        const serverCategories = categoryList.categories;
        if (serverCategories.length) {
            const genreFacetValues = serverCategories
                .filter((category) => {
                return serverData.isDefinedNonNull(category.genreId);
            })
                .map((category) => {
                return new models.SearchFacetValue(category.name, category.genreId, selectedFacets["genre"]);
            });
            genreFacetValues.unshift(new models.SearchFacetValue(objectGraph.loc.string("SEARCH_FACET_ANY_CATEGORY", "Any"), null, selectedFacets["genre"]));
            facets.push(new models.SearchFacetSet("genre", genreFacetValues));
        }
    }
    const searchSortOptions = objectGraph.bag.searchSortOptions;
    // Sorts
    const sortFacetValues = [];
    sortFacetValues.push(new models.SearchFacetValue(objectGraph.loc.string("SEARCH_FACET_RELEVANCE"), null, selectedFacets["sort"]));
    for (const sortValue of searchSortOptions) {
        sortFacetValues.push(new models.SearchFacetValue(objectGraph.loc.string("SEARCH_FACET_" + sortValue), sortValue, selectedFacets["sort"]));
    }
    if (sortFacetValues.length > 1) {
        facets.push(new models.SearchFacetSet("sort", sortFacetValues));
    }
    const serverAgeBands = objectGraph.bag.ageBands;
    const ageBandFacetValues = serverAgeBands.map((ageBand) => {
        return new models.SearchFacetValue(serverData.asString(ageBand, "name"), serverData.asString(ageBand, "ageBandId"), selectedFacets["ages"]);
    });
    if (ageBandFacetValues.length > 0 && objectGraph.client.deviceType !== "mac") {
        facets.push(new models.SearchFacetSet("ages", ageBandFacetValues));
    }
    return facets;
}
/**
 * Create Search Facets. Theres platform specific variations here.
 * @param categoryFacetsData Additional
 */
export function createSearchPageFacets(objectGraph, categoryFacetsData) {
    let categoryFacet = null;
    let sortsFacet = null;
    let ageBandsFacet = null;
    // Platform
    const deviceTypeFacet = new models.PageFacetsFacet("targetPlatform", "targetPlatform", objectGraph.loc.string("SEARCH_FACET_TYPE_TITLE_DEVICE_TYPE"), "singleSelection", [
        new models.PageFacetOption(objectGraph.loc.string("Search.Facets.iPadAndIPhone"), null),
        new models.PageFacetOption(objectGraph.loc.string("Search.Facets.iPhoneOnly"), "iphone"),
    ], null, null, pageFacetChangeAction(objectGraph, "targetPlatform"));
    // Price
    const priceFacet = new models.PageFacetsFacet("filter[price]", "filter[price]", objectGraph.loc.string("SEARCH_FACET_TYPE_TITLE_PRICE"), "singleSelection", [
        new models.PageFacetOption(objectGraph.loc.string("SEARCH_FACET_ANY_PRICE", "Any"), null),
        new models.PageFacetOption(objectGraph.loc.string("SEARCH_FACET_FREE", "Any"), "free"),
    ], null, null, pageFacetChangeAction(objectGraph, "price"));
    // Categories
    const categoryList = categoryListFromApiResponse(objectGraph, categoryFacetsData, false);
    if (categoryList) {
        const serverCategories = categoryList.categories;
        if (serverCategories.length) {
            const categories = serverCategories.filter((category) => {
                return serverData.isDefinedNonNull(category.genreId);
            });
            categoryFacet = new models.PageFacetsFacet("filter[genre]", "filter[genre]", objectGraph.loc.string("SEARCH_FACET_TYPE_TITLE_CATEGORY"), "singleSelection", [new models.PageFacetOption(objectGraph.loc.string("SEARCH_FACET_ANY_CATEGORY", "Any"), null)], null, null, pageFacetChangeAction(objectGraph, "genre"));
            for (const category of categories) {
                categoryFacet.options.push(new models.PageFacetOption(category.name, category.genreId));
            }
        }
    }
    // Sorts
    const searchSortOptions = objectGraph.bag.searchSortOptions;
    sortsFacet = new models.PageFacetsFacet("sort", "sort", objectGraph.loc.string("SEARCH_FACET_TYPE_TITLE_SORT"), "singleSelection", [new models.PageFacetOption(objectGraph.loc.string("SEARCH_FACET_RELEVANCE"), null)], null, null, pageFacetChangeAction(objectGraph, "sort"));
    for (const sortValue of searchSortOptions) {
        sortsFacet.options.push(new models.PageFacetOption(objectGraph.loc.string("SEARCH_FACET_" + sortValue), sortValue));
    }
    // Age Bands
    const serverAgeBands = objectGraph.bag.ageBands;
    const ageBandFacetOptions = serverAgeBands.map((ageBand) => {
        return new models.PageFacetOption(serverData.asString(ageBand, "name"), serverData.asString(ageBand, "ageBandId"));
    });
    if (ageBandFacetOptions.length > 0 && objectGraph.client.deviceType !== "mac") {
        ageBandsFacet = new models.PageFacetsFacet("filter[ages]", "filter[ages]", objectGraph.loc.string("SEARCH_FACET_TYPE_TITLE_AGE_BAND"), "singleSelection", ageBandFacetOptions, null, null, pageFacetChangeAction(objectGraph, "ages"));
    }
    const pageFacets = new models.PageFacets([], false, null);
    if (objectGraph.client.isMac) {
        pageFacets.facetGroups.push(new models.PageFacetsGroup([priceFacet]));
        if (serverData.isDefinedNonNull(categoryFacet)) {
            pageFacets.facetGroups.push(new models.PageFacetsGroup([categoryFacet]));
        }
        pageFacets.facetGroups.push(new models.PageFacetsGroup([sortsFacet]));
    }
    else {
        const facets = [deviceTypeFacet, priceFacet];
        if (serverData.isDefinedNonNull(categoryFacet)) {
            facets.push(categoryFacet);
        }
        facets.push(sortsFacet);
        if (serverData.isDefinedNonNull(ageBandsFacet)) {
            facets.push(ageBandsFacet);
        }
        for (const facet of facets) {
            facet.showsSelectedOptions = true;
        }
        pageFacets.facetGroups.push(new models.PageFacetsGroup(facets));
    }
    return pageFacets;
}
export function createDefaultSelectedFacetOptions(objectGraph) {
    return {
        "targetPlatform": [new models.PageFacetOption(objectGraph.loc.string("SEARCH_FACET_IPAD_ONLY"), null)],
        "filter[price]": [new models.PageFacetOption(objectGraph.loc.string("SEARCH_FACET_ANY_PRICE", "Any"), null)],
        "sort": [new models.PageFacetOption(objectGraph.loc.string("SEARCH_FACET_RELEVANCE"), null)],
        "filter[genre]": [new models.PageFacetOption(objectGraph.loc.string("SEARCH_FACET_ANY_CATEGORY", "Any"), null)],
    };
}
function pageFacetChangeAction(objectGraph, facetParameter) {
    const action = new models.BlankAction();
    metricsClickHelpers.addClickEventToPageFacetsChangeAction(objectGraph, action, facetParameter);
    return action;
}
//# sourceMappingURL=search-facets.js.map