/**
 * Model Builder for Guided Search (STUB)
 */
import { GuidedSearchQuery, GuidedSearchToken, SearchAction, searchEntitySystemImage, } from "../../../api/models";
import { GuidedSearchTokenToggleAction, SearchEntityChangeAction, } from "../../../api/models/search/guided-search-actions";
import { isNullOrEmpty } from "../../../foundation/json-parsing/server-data";
import { unreachable } from "../../../foundation/util/errors";
import { addEventsToGuidedSearchTokenEntityChangeAction, addEventsToGuidedSearchTokenAction, addImpressionMetricsToGuidedSearchToken, } from "./guided-search-metrics";
// region exports
/**
 * Create a Guided search token from facet data.
 * @param objectGraph The App Store object graph.
 * @param selectionBehavior The behavior for click action, e.g. query rewrite versus toggling tokens.
 * @param requestDescriptor The request descriptor for search request that returned this data.
 * @param facetData The facet data to build with.
 * @param metricsOptions The metrics options.
 */
export function createGuidedSearchToken(objectGraph, selectionBehavior, requestDescriptor, facetData, metricsOptions) {
    if (isNullOrEmpty(facetData)) {
        return null;
    }
    // Create click and search metrics for token action.
    const origin = "guidedToken";
    const searchTerm = requestDescriptor.term;
    const targetToken = facetData.displayLabel;
    const clickAction = selectionBehavior === "rewrite"
        ? new SearchAction(targetToken, facetData.finalTerm, null, origin)
        : new GuidedSearchTokenToggleAction(targetToken, origin);
    addEventsToGuidedSearchTokenAction(objectGraph, clickAction, targetToken, searchTerm, metricsOptions);
    // Create the token with associated click action and metrics..
    const token = new GuidedSearchToken(targetToken, facetData.isSelected, undefined, targetToken, clickAction);
    addImpressionMetricsToGuidedSearchToken(objectGraph, token, "guidedLabel", metricsOptions);
    return token;
}
/**
 * Create the guided search queries from facet data.
 * These are stored across a guided search session so client can send down precomputed combinations of search term and guided search facets as an optimization.
 * @param requestDescriptor The request descriptor for search request that returned this data.
 * @param facetData The data to generate `GuidedSearchQuery` for.
 */
export function createGuidedSearchQueries(objectGraph, requestDescriptor, facetData) {
    var _a;
    if (isNullOrEmpty(facetData)) {
        return null;
    }
    // request parameters that returned this `facetData`
    const requestTerm = requestDescriptor.term;
    const requestFacets = (_a = requestDescriptor.guidedSearchTokens) !== null && _a !== void 0 ? _a : [];
    const queries = [];
    for (const data of facetData) {
        /**
         * For each facet data, project the facet selection if that facet was selected / deselected w.r.t. request's facets
         */
        const facetValue = data.displayLabel;
        const lookaheadFacets = Array.from(requestFacets);
        if (data.isSelected) {
            const facetIndex = lookaheadFacets.indexOf(facetValue);
            if (facetIndex !== -1) {
                lookaheadFacets.splice(facetIndex, 1);
            }
        }
        else {
            lookaheadFacets.push(facetValue);
        }
        const query = new GuidedSearchQuery(requestTerm, lookaheadFacets, data.finalTerm);
        queries.push(query);
    }
    return queries;
}
/**
 * Create an action for **reversing** a selected entity hint. This is so user can reverse an selected entity filter for search.
 * e.g. If user tapped "Search for Games in Apple Watch Apps", user can deselect the "Apple Watch Apps" entity filter from the Guided Search UI
 */
export function createGuidedSearchTokenClearingEntityFilter(objectGraph, requestDescriptor, metricsOptions) {
    var _a;
    const selectedEntityFilter = requestDescriptor.searchEntity;
    if (!selectedEntityFilter) {
        return null;
    }
    const deselectEntityAction = new SearchEntityChangeAction(null, "guidedToken");
    addEventsToGuidedSearchTokenEntityChangeAction(objectGraph, deselectEntityAction, requestDescriptor.term, selectedEntityFilter, metricsOptions);
    let displayText;
    switch (selectedEntityFilter) {
        case "arcade":
            displayText = objectGraph.loc.string("GUIDED_SEARCH_TOKEN_ENTITY_ARCADE");
            break;
        case "developer":
            displayText = objectGraph.loc.string("GUIDED_SEARCH_TOKEN_ENTITY_DEVELOPERS");
            break;
        case "story":
            displayText = objectGraph.loc.string("GUIDED_SEARCH_TOKEN_ENTITY_STORIES");
            break;
        case "watch":
            displayText = objectGraph.loc.string("GUIDED_SEARCH_TOKEN_ENTITY_APPLEWATCH");
            break;
        default:
            unreachable(selectedEntityFilter);
            break;
    }
    const token = new GuidedSearchToken(displayText, true, (_a = searchEntitySystemImage(selectedEntityFilter)) !== null && _a !== void 0 ? _a : "magnifyingglass", displayText, deselectEntityAction);
    addImpressionMetricsToGuidedSearchToken(objectGraph, token, "hintsEntity", metricsOptions);
    return token;
}
// endregion
//# sourceMappingURL=guided-search.js.map