import * as validation from "@jet/environment/json/validation";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as contentAttributes from "../../content/attributes";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as metricsHelpersPage from "../../metrics/helpers/page";
import { isSome } from "@jet/environment";
/**
 * Create a shelf for the product page's version history.
 *
 * @param data The raw data response for a product page JSON fetch.
 * @param shelfMetrics The product page shelf metrics.
 * @param shelfContext A collection of any other variables used when creating this shelf.
 * @returns A version history shelf.
 */
export function create(objectGraph, data, shelfMetrics, shelfContext) {
    return validation.context("create", () => {
        return standardMostRecentVersionShelf(objectGraph, data, shelfContext.isFirstPartyHideableApp, shelfContext.isBundle, shelfContext.isPreorder, shelfMetrics);
    });
}
/**
 * Create a shelf for the product page description.
 *
 * @param data The raw data response for a product page JSON fetch.
 * @param isFirstPartyHideableApp Indicates whether this app is first party and/or hideable.
 * @param isBundle Whether this is a bundle id
 * @param isPreorder Whether the buy is for a preorder.
 * @returns A description shelf.
 */
export function mostRecentTitledParagraphFromVersionHistory(objectGraph, data, isFirstPartyHideableApp, isBundle, isPreorder) {
    if (!isFirstPartyHideableApp && !isBundle && !isPreorder) {
        const versions = contentAttributes.contentAttributeAsArrayOrEmpty(objectGraph, data, "versionHistory");
        if (versions.length > 1) {
            const versionHistory = versionHistoryFromApiResponse(objectGraph, versions);
            if (versionHistory.length > 0) {
                return titledParagraphFromVersionHistoryEntry(objectGraph, versions[0], "overview");
            }
        }
    }
    return null;
}
/**
 * A flow action to navigate to the version history.
 *
 * @param versions The raw data response for the version history of a product.
 * @param productId The product ID.
 * @returns A flow action for navigating to the version history.
 */
export function versionHistoryPageAction(objectGraph, versions, productId) {
    return validation.context("versionHistoryPageAction", () => {
        const versionHistory = versionHistoryFromApiResponse(objectGraph, versions);
        let action;
        if (versionHistory.length > 0) {
            // Create version history page
            const allVersionHistoryShelf = new models.Shelf("titledParagraph");
            allVersionHistoryShelf.items = versionHistory;
            const page = new models.GenericPage([allVersionHistoryShelf]);
            page.title = objectGraph.loc.string("VERSION_HISTORY_PAGE_TITLE", "Version History");
            if (objectGraph.client.deviceType !== "watch") {
                page.presentationOptions = ["prefersLargeTitle"];
            }
            const pageInformation = metricsHelpersPage.fakeMetricsPageInformation(objectGraph, "VersionHistory", productId, "");
            metricsHelpersPage.addMetricsEventsToPageWithInformation(objectGraph, page, pageInformation);
            // Create version history action
            const versionHistorySeeAll = new models.FlowAction("versionHistory");
            versionHistorySeeAll.title = objectGraph.loc.string("ACTION_VERSION_HISTORY_SEE_ALL");
            versionHistorySeeAll.pageData = page;
            action = versionHistorySeeAll;
        }
        return action;
    });
}
function standardMostRecentVersionShelf(objectGraph, productData, isFirstPartyHideableApp, isBundle, isPreorder, shelfMetrics) {
    return validation.context("standardMostRecentVersionShelf", () => {
        if (!isFirstPartyHideableApp && !isBundle && !isPreorder) {
            const versions = contentAttributes.contentAttributeAsArrayOrEmpty(objectGraph, productData, "versionHistory");
            if (versions.length > 1) {
                const versionHistory = versionHistoryFromApiResponse(objectGraph, versions);
                if (versionHistory.length > 0) {
                    const shelf = new models.Shelf("titledParagraph");
                    shelf.title = objectGraph.loc.string("ProductPage.Section.MostRecentVersion.Title");
                    // Add recent version
                    const recentVersion = titledParagraphFromVersionHistoryEntry(objectGraph, versions[0], "overview");
                    shelf.items = [recentVersion];
                    shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "versionHistory");
                    if (shelf) {
                        if (objectGraph.client.deviceType !== "watch") {
                            shelf.seeAllAction = versionHistorySeeAllAction(objectGraph, versions, productData.id, shelfMetrics);
                        }
                        return shelf;
                    }
                }
            }
        }
        return null;
    });
}
function versionHistorySeeAllAction(objectGraph, versions, productId, shelfMetrics) {
    return validation.context("versionHistorySeeAllAction", () => {
        const action = versionHistoryPageAction(objectGraph, versions, productId);
        metricsHelpersClicks.addClickEventToSeeAllAction(objectGraph, action, null, {
            pageInformation: shelfMetrics.metricsPageInformation,
            locationTracker: shelfMetrics.locationTracker,
        });
        return action;
    });
}
/**
 *  Create an array of product releases from an array of version history data objects.
 *  @param versions    The data objects to convert.
 *  @return An array of titled paragraph model objects.
 */
function versionHistoryFromApiResponse(objectGraph, versions) {
    if (!versions) {
        return [];
    }
    return versions.map((version) => {
        return titledParagraphFromVersionHistoryEntry(objectGraph, version, "detail");
    });
}
function titledParagraphFromVersionHistoryEntry(objectGraph, versionEntry, style) {
    return validation.context("titledParagraphFromVersionHistoryEntry", () => {
        // Notes
        let releaseNotes = serverData.asString(versionEntry, "releaseNotes");
        if (isSome(releaseNotes)) {
            releaseNotes = releaseNotes.trim();
        }
        const release = new models.TitledParagraph(releaseNotes, style);
        // Version
        const versionString = serverData.asString(versionEntry, "versionDisplay");
        const versionTemplate = objectGraph.loc.string("VERSION_STRING_TEMPLATE");
        switch (style) {
            case "detail":
                release.primarySubtitle = versionString;
                break;
            default:
                release.primarySubtitle = versionTemplate.replace("{version}", versionString);
                break;
        }
        // Release date
        const releaseDateString = serverData.asString(versionEntry, "releaseTimestamp");
        if (releaseDateString) {
            const releaseDate = new Date(releaseDateString);
            release.secondarySubtitle = objectGraph.loc.relativeDate(releaseDate);
        }
        return release;
    });
}
//# sourceMappingURL=version-history-shelves.js.map