import * as validation from "@jet/environment/json/validation";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as mediaRelationship from "../../../foundation/media/relationships";
import { shallowCopyOf } from "../../../foundation/util/objects";
import * as contentAttributes from "../../content/attributes";
import { developerUrlFromDeveloperData } from "../../content/content";
import * as lockups from "../../lockups/lockups";
import { addClickEventToAction } from "../../metrics/helpers/clicks";
import { addMetricsEventsToPageWithInformation, fakeMetricsPageInformation } from "../../metrics/helpers/page";
import * as badgesCommon from "../badges/badges-common";
import * as descriptionShelf from "./description-shelf";
import * as versionHistoryShelves from "./version-history-shelves";
const tvTextCardList = [aboutTextCard, editorsNoteTextCard, whatsNewTextCard];
const visionTextCardList = [aboutTextCard, whatsNewTextCard];
/**
 * Create a shelf for the text cards displayed on the tv product page.
 *
 * @param data The raw data response for a product page JSON fetch.
 * @param shelfMetrics The product page shelf metrics.
 * @returns A textcard shelf.
 */
export function create(objectGraph, data, shelfMetrics) {
    return validation.context("textCardShelfFromResponse", () => {
        if (serverData.isNullOrEmpty(data)) {
            return null;
        }
        let textCardList;
        if (objectGraph.client.isTV) {
            textCardList = tvTextCardList;
        }
        else if (objectGraph.client.isVision) {
            textCardList = visionTextCardList;
        }
        else {
            return null;
        }
        const shelf = new models.Shelf("textCard");
        if (!objectGraph.client.isVision) {
            shelf.title = objectGraph.loc.string("PRODUCT_SECTION_TEXT_CARDS");
            shelf.isHorizontal = true;
        }
        shelf.items = createTextCards(objectGraph, textCardList, data, shelfMetrics);
        if (objectGraph.client.isVision && shelf.items.length === 1) {
            shelf.presentationHints = {
                isSingleDensity: true,
            };
        }
        shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "links");
        return shelf;
    });
}
/**
 * Creates a text card object
 *
 * @param objectGraph The current object graph
 * @param style The style of the text card
 * @param title The title text
 * @param subtitle The subtitle text
 * @param descriptionText The description text
 * @param alertTitle The title to show when presenting as an alert
 * @param captionTitle The caption title text
 * @param captionSubtitle The caption subtitle text
 * @param captionArtwork The artwork to display alongside caption text
 * @param includeCaptionWhenExpanded Whether to display the caption when in an expanded state
 * @param captionAction An action to run when selecting the caption
 * @returns A built text card
 */
function createTextCard(objectGraph, style, title, subtitle, descriptionText, alertTitle, captionTitle, captionSubtitle, captionArtwork, includeCaptionWhenExpanded, captionAction) {
    const textCard = new models.TextCard(style, title, subtitle);
    if (objectGraph.host.isTV || objectGraph.client.isVision) {
        const bodyParagraph = new models.Paragraph(descriptionText);
        if (objectGraph.featureFlags.isEnabled("tv_product_page_2024E")) {
            bodyParagraph.style = "unadorned";
        }
        bodyParagraph.isCollapsed = true;
        textCard.bodyParagraph = bodyParagraph;
    }
    const paragraph = new models.Paragraph(descriptionText);
    paragraph.wantsCollapsedNewlines = false;
    if (objectGraph.client.isVision) {
        textCard.captionTitle = captionTitle;
        textCard.captionSubtitle = captionSubtitle;
        textCard.captionArtwork = captionArtwork;
        textCard.includeCaptionWhenExpanded = includeCaptionWhenExpanded;
        textCard.captionAction = captionAction;
    }
    else {
        textCard.regularAction = new models.ScrollingAlertAction(alertTitle, paragraph);
    }
    return textCard;
}
function aboutTextCard(objectGraph, data, metricsContext) {
    const paragraph = descriptionShelf.paragraphFromData(objectGraph, data);
    if (serverData.isNull(paragraph)) {
        return null;
    }
    const descriptionText = paragraph.text;
    if (objectGraph.client.isVision) {
        const title = objectGraph.loc.string("PRODUCT_SECTION_DESCRIPTION");
        const captionTitle = objectGraph.loc.string("PRODUCT_DEVELOPER_TITLE");
        const captionSubtitle = mediaAttributes.attributeAsString(data, "artistName");
        const developer = mediaRelationship.relationshipData(objectGraph, data, "developer");
        const developerArtworkData = mediaAttributes.attributeAsDictionary(developer, "editorialArtwork.brandLogo");
        const developerArtwork = badgesCommon.badgeArtwork(objectGraph, developerArtworkData);
        // Developer action
        let captionAction;
        const developerUrl = developerUrlFromDeveloperData(objectGraph, developer);
        if (serverData.isDefinedNonNull(developerUrl)) {
            const developerAction = new models.FlowAction("page");
            developerAction.title = mediaAttributes.attributeAsString(data, "artistName");
            developerAction.pageUrl = developerUrl;
            const flowBackAction = new models.FlowBackAction("sheetDismiss");
            captionAction = new models.CompoundAction([flowBackAction, developerAction]);
            addClickEventToAction(objectGraph, captionAction, {
                targetType: "button",
                id: developer.id,
                actionType: "navigate",
                pageInformation: metricsContext.pageInformation,
                locationTracker: metricsContext.locationTracker,
            });
        }
        const textCard = createTextCard(objectGraph, "text", title, null, descriptionText, title, captionTitle, captionSubtitle, developerArtwork, true, captionAction);
        textCard.regularAction = createTextCardDetailPageFlowAction(objectGraph, textCard, data.id);
        return textCard;
    }
    else {
        const title = mediaAttributes.attributeAsString(data, "name");
        const subtitle = lockups.subtitleFromData(objectGraph, data);
        return createTextCard(objectGraph, "text", title, subtitle, descriptionText, title);
    }
}
function editorsNoteTextCard(objectGraph, data, metricsContext) {
    const editorialNotes = contentAttributes.contentAttributeAsDictionary(objectGraph, data, "editorialNotes");
    const editorsNotes = serverData.asString(editorialNotes, "standard");
    if (editorsNotes) {
        let title = null;
        let alertTitle = null;
        let style = "text";
        const editorialBadgeInfo = contentAttributes.contentAttributeAsDictionary(objectGraph, data, "editorialBadgeInfo");
        if (editorialBadgeInfo) {
            const badgeType = serverData.asString(editorialBadgeInfo, "editorialBadgeType");
            if (badgeType && badgeType === "editorialPriority") {
                style = "editorsChoice";
            }
            const alertTitleFromServer = serverData.asString(editorialBadgeInfo, "nameForDisplay");
            if (alertTitleFromServer) {
                alertTitle = alertTitleFromServer;
            }
            else {
                alertTitle = objectGraph.loc.string("APP_STORE_EDITORS_NOTES");
            }
        }
        else {
            title = objectGraph.loc.string("APP_STORE_EDITORS_NOTES");
            alertTitle = objectGraph.loc.string("APP_STORE_EDITORS_NOTES");
        }
        return createTextCard(objectGraph, style, title, "", editorsNotes, alertTitle);
    }
    return null;
}
function whatsNewTextCard(objectGraph, data, metricsContext) {
    const isBundle = data.type === "app-bundles";
    const isFirstPartyHideableApp = mediaAttributes.attributeAsBooleanOrFalse(data, "isFirstPartyHideableApp");
    const isPreorder = mediaAttributes.attributeAsBoolean(data, "isPreorder");
    const mostRecentVersionNotes = versionHistoryShelves.mostRecentTitledParagraphFromVersionHistory(objectGraph, data, isFirstPartyHideableApp, isBundle, isPreorder);
    if (mostRecentVersionNotes) {
        const versionNumber = mostRecentVersionNotes.primarySubtitle;
        const releaseNotes = mostRecentVersionNotes.text;
        const title = objectGraph.loc.string("ProductPage.Section.MostRecentVersion.Title");
        if (objectGraph.client.isVision) {
            const releaseDate = mostRecentVersionNotes.secondarySubtitle;
            const textCard = createTextCard(objectGraph, "text", title, null, releaseNotes, title, releaseDate, versionNumber);
            const versions = contentAttributes.contentAttributeAsArrayOrEmpty(objectGraph, data, "versionHistory");
            if (versions.length > 0 && !isFirstPartyHideableApp && !isBundle && !isPreorder) {
                textCard.regularAction = versionHistoryShelves.versionHistoryPageAction(objectGraph, versions, data.id);
            }
            return textCard;
        }
        else {
            return createTextCard(objectGraph, "text", title, versionNumber, releaseNotes, title);
        }
    }
    return null;
}
function createTextCards(objectGraph, textCardList, data, shelfMetrics) {
    const textCards = [];
    const metricsContext = {
        locationTracker: shelfMetrics.locationTracker,
        pageInformation: shelfMetrics.metricsPageInformation,
    };
    for (const f of textCardList) {
        const textCard = f(objectGraph, data, metricsContext);
        if (textCard) {
            textCards.push(textCard);
        }
    }
    return textCards;
}
/**
 * Creates a flow action for navigating to the detail page for a text card.
 *
 * @param objectGraph The current object graph
 * @param textCard The source text card
 * @param productId The id of the owning product
 * @returns A flow action
 */
function createTextCardDetailPageFlowAction(objectGraph, textCard, productId) {
    const detailTextCard = shallowCopyOf(textCard);
    detailTextCard.isExpanded = true;
    const shelf = new models.Shelf("textCard");
    shelf.items = [detailTextCard];
    shelf.presentationHints = {
        isSingleDensity: true,
    };
    const page = new models.GenericPage([shelf]);
    page.title = textCard.title;
    const pageInformation = fakeMetricsPageInformation(objectGraph, "DescriptionDetails", productId, "");
    addMetricsEventsToPageWithInformation(objectGraph, page, pageInformation);
    const flowAction = new models.FlowAction("textCardDetail");
    flowAction.title = textCard.title;
    flowAction.pageData = page;
    return flowAction;
}
//# sourceMappingURL=textcard-shelf.js.map