import * as validation from "@jet/environment/json/validation";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as contentAttributes from "../../content/attributes";
import * as productPageUtil from "../product-page-util";
/**
 * Creates a shelf of text links
 * @param data The raw data response for a product page JSON fetch.
 * @param shelfMetrics The current product page shelf metrics
 * @returns The shelf of text links. If objectGraph.client.deviceType is not tv, returns null.
 */
export function create(objectGraph, data, shelfMetrics) {
    return validation.context("create", () => {
        if (objectGraph.client.deviceType !== "tv") {
            return null;
        }
        if (serverData.isNullOrEmpty(data)) {
            return null;
        }
        const shelf = new models.Shelf("paragraph");
        shelf.isHorizontal = false;
        const links = externalProductLinksAsParagraphsFromData(objectGraph, data);
        if (links && links.length > 0) {
            shelf.items = links;
        }
        if (productPageUtil.isShelfBased(objectGraph)) {
            shelf.background = {
                type: "darkOverlay",
            };
        }
        shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "links");
        return shelf;
    });
}
/**
 * Product links displayable as text
 * @param data The product data to use
 * @returns {models.Paragraph[]} Product links displayed as paragraphs
 */
function externalProductLinksAsParagraphsFromData(objectGraph, data) {
    if (!data) {
        return null;
    }
    return validation.context("externalProductLinksAsParagraphsFromData", () => {
        const links = [];
        const developerWebsiteUrl = contentAttributes.contentAttributeAsString(objectGraph, data, "websiteUrl");
        if (developerWebsiteUrl) {
            const text = objectGraph.loc.string("DEVELOPER_WEBSITE_WITH_URL").replace("{URL}", developerWebsiteUrl);
            const paragraph = new models.Paragraph(text);
            if (objectGraph.featureFlags.isEnabled("tv_product_page_2024E")) {
                paragraph.style = "unadorned";
            }
            paragraph.alignment = "center";
            links.push(paragraph);
        }
        return links;
    });
}
//# sourceMappingURL=text-links-shelf.js.map