import * as validation from "@jet/environment/json/validation";
import * as models from "../../../api/models";
import { isDefinedNonNullNonEmpty, isNullOrEmpty } from "../../../foundation/json-parsing/server-data";
import * as metricsHelpersLocation from "../../metrics/helpers/location";
import { insertPlaceholdersIntoProductPageShelfIfRequired } from "../../placeholders/placeholders";
import * as todayHorizontalCardUtil from "../../today/today-horizontal-card-util";
import { TodayParseContext } from "../../today/today-types";
import * as productPageCommon from "../product-page-common";
import { ProductPageOnDemandShelfType } from "../product-page-common";
import { ProductPageShelfMetrics } from "../product-page-shelf-metrics";
/**
 * Create a shelf for the product page's small story cards.
 *
 * @param productData The raw product data response for a product page JSON fetch.
 * @param data The raw data items for a product page JSON fetch.
 * @param shelfMetrics The product page shelf metrics.
 * @returns A small story card shelf.
 */
export function createInitialShelf(objectGraph, productData, data, shelfMetrics) {
    return validation.context("smallStoryShelf", () => {
        var _a;
        if (isNullOrEmpty(data)) {
            return null;
        }
        const metricsOptions = {
            pageInformation: shelfMetrics.metricsPageInformation,
            locationTracker: shelfMetrics.locationTracker,
            targetType: "swoosh",
            id: `${shelfMetrics.getSequenceId()}`,
            idType: "sequential",
        };
        const title = objectGraph.loc.string("ProductPage.Section.FeaturedIn.Title");
        metricsHelpersLocation.pushContentLocation(objectGraph, metricsOptions, title);
        const remainingItems = [];
        const cardUnavailable = function (cardData) {
            remainingItems.push(cardData);
            return false;
        };
        let shelf;
        if (areMiniTodayCardsEnabled(objectGraph)) {
            const todayParseContext = new TodayParseContext(metricsOptions.pageInformation, metricsOptions.locationTracker);
            shelf = todayHorizontalCardUtil.shelfForMiniTodayCards(objectGraph, data, title, null, todayParseContext, cardUnavailable);
        }
        else {
            const context = {
                metricsLocationTracker: metricsOptions.locationTracker,
                metricsPageInformation: metricsOptions.pageInformation,
                filterOutMediaCardKinds: new Set(["appIcon"]), // policy decision to filter out these types on the featured in shelf of a product page.
            };
            shelf = todayHorizontalCardUtil.shelfForHorizontalCardItems(objectGraph, data, "smallStoryCard", title, null, context, cardUnavailable);
            if (Array.isArray(shelf.items)) {
                shelf.items = shelf.items.filter((item) => {
                    if (!(item instanceof models.TodayCard)) {
                        return true;
                    }
                    return todayHorizontalCardUtil.isHorizontalCardSupportedForKind(objectGraph, item.media.kind, "smallStoryCard");
                });
            }
        }
        shelf.isHorizontal = true;
        if (!((_a = shelf.items) === null || _a === void 0 ? void 0 : _a.length) && remainingItems.length === 0) {
            return null;
        }
        if (objectGraph.client.isiOS && objectGraph.bag.isOnDemandShelfFetchingEnabled) {
            shelf.fetchStrategy = models.IncompleteShelfFetchStrategy.OnShelfWillAppear;
        }
        if (isDefinedNonNullNonEmpty(remainingItems)) {
            metricsHelpersLocation.popLocation(shelfMetrics.locationTracker);
            shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "featuredIn");
            metricsHelpersLocation.nextPosition(shelfMetrics.locationTracker);
            const shelfToken = new productPageCommon.ProductPageShelfToken(productData.id, remainingItems, shelf.title, false, undefined, shelf.contentType, null, null, null, null, null, ProductPageOnDemandShelfType.SmallStory);
            insertPlaceholdersIntoProductPageShelfIfRequired(objectGraph, shelf, shelfToken);
            shelf.url = productPageCommon.shelfContentUrl(objectGraph, shelfToken, shelfMetrics);
        }
        else {
            metricsHelpersLocation.popLocation(shelfMetrics.locationTracker);
            shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "featuredIn");
            metricsHelpersLocation.nextPosition(shelfMetrics.locationTracker);
        }
        return shelf;
    });
}
/**
 * Create a shelf of small store card items for the secondary look up.
 *
 * @param dataItems The raw data items from the secondary fetch.
 * @param shelfToken The shelf token for this shelf.
 * @returns A small story card items shelf.
 */
export function createSecondaryShelf(objectGraph, dataItems, shelfToken) {
    return validation.context("smallStorySecondaryShelf", () => {
        if (isNullOrEmpty(dataItems)) {
            return null;
        }
        const shelfMetrics = new ProductPageShelfMetrics(shelfToken.sourcePageInformation, shelfToken.sourceLocationTracker, shelfToken.sourceSequenceId);
        const metricsOptions = {
            pageInformation: shelfMetrics.metricsPageInformation,
            locationTracker: shelfMetrics.locationTracker,
            targetType: "swoosh",
            id: `${shelfMetrics.getSequenceId()}`,
            idType: "sequential",
            excludeAttribution: true,
            recoMetricsData: shelfToken.recoMetricsData,
        };
        metricsHelpersLocation.pushContentLocation(objectGraph, metricsOptions, shelfToken.title);
        let shelf;
        if (areMiniTodayCardsEnabled(objectGraph)) {
            const todayParseContext = new TodayParseContext(metricsOptions.pageInformation, metricsOptions.locationTracker);
            shelf = todayHorizontalCardUtil.shelfForMiniTodayCards(objectGraph, dataItems !== null && dataItems !== void 0 ? dataItems : [], shelfToken.title, null, todayParseContext);
        }
        else {
            const context = {
                metricsLocationTracker: metricsOptions.locationTracker,
                metricsPageInformation: metricsOptions.pageInformation,
                filterOutMediaCardKinds: productPageCommon.filteredMediaCardKindsForSmallStoryCardOnPlatform(objectGraph.host.platform),
            };
            shelf = todayHorizontalCardUtil.shelfForHorizontalCardItems(objectGraph, dataItems, "smallStoryCard", shelfToken.title, null, context);
            if (Array.isArray(shelf === null || shelf === void 0 ? void 0 : shelf.items)) {
                shelf.items = shelf.items.filter((item) => {
                    if (!(item instanceof models.TodayCard)) {
                        return true;
                    }
                    return todayHorizontalCardUtil.isHorizontalCardSupportedForKind(objectGraph, item.media.kind, "smallStoryCard");
                });
            }
        }
        shelf.isHorizontal = true;
        shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "featuredIn");
        metricsHelpersLocation.popLocation(shelfMetrics.locationTracker);
        metricsHelpersLocation.nextPosition(shelfMetrics.locationTracker);
        return shelf;
    });
}
function areMiniTodayCardsEnabled(objectGraph) {
    return objectGraph.client.isiOS && objectGraph.featureFlags.isEnabled("mini_today_cards_product");
}
//# sourceMappingURL=small-story-card-shelf.js.map