import * as validation from "@jet/environment/json/validation";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as mediaDataStructure from "../../../foundation/media/data-structure";
import * as mediaRelationship from "../../../foundation/media/relationships";
import { shallowCopyOf } from "../../../foundation/util/objects";
import { createArtworkForResource } from "../../content/artwork/artwork";
import * as contentAttributes from "../../content/attributes";
import * as content from "../../content/content";
import * as reviews from "../reviews";
import { isNothing, isSome } from "@jet/environment";
/**
 * Create a shelf for the product page reviews.
 *
 * @param objectGraph
 * @param productData
 * @param shelfMetrics The product page shelf metrics.
 * @param shelfContext A collection of any other variables used when creating this shelf.
 * @returns A reviews shelf.
 */
export function createReviewsShelves(objectGraph, productData, shelfMetrics, shelfContext, lockupSubtitle) {
    return validation.context("createReviewsShelves", () => {
        const productReviewsShelves = {};
        if (objectGraph.client.isWatch) {
            return productReviewsShelves;
        }
        if (serverData.isNullOrEmpty(productData)) {
            return productReviewsShelves;
        }
        const isReviewable = reviews.isAppReviewable(objectGraph, productData);
        const appPlatforms = content.supportedAppPlatformsFromData(objectGraph, productData);
        const reviewsData = mediaRelationship.relationship(productData, "reviews");
        const reviewItems = mediaDataStructure.dataCollectionFromDataContainer(reviewsData);
        const ratingsData = mediaAttributes.attributeAsDictionary(productData, "userRating");
        ratingsData.ratingAverage = serverData.asNumber(ratingsData, "value");
        ratingsData.adamId = productData.id;
        ratingsData.isBundle = shelfContext.isBundle;
        ratingsData.supportUrl = contentAttributes.contentAttributeAsString(objectGraph, productData, "supportURLForLanguage");
        const appIcon = content.iconFromData(objectGraph, productData, {
            useCase: 2 /* content.ArtworkUseCase.LockupIconMedium */,
        });
        const tvOnlyApp = appPlatforms.length === 1 && appPlatforms[0] === "tv";
        const reviewsTitle = reviews.sectionTitleForPlatform(objectGraph);
        const productRatingsWereReset = serverData.asBooleanOrFalse(ratingsData, "wasReset");
        const productRatings = createProductRatings(objectGraph, ratingsData, reviewItems.length, productRatingsWereReset);
        const tapToRateProductReviewAction = createTapToRateProductReviewAction(objectGraph, productData, shelfContext.productTitle, ratingsData, shelfContext.isBundle, tvOnlyApp);
        const writeAReview = createWriteAReview(objectGraph, productData, ratingsData, tvOnlyApp, appIcon, lockupSubtitle, shelfContext.productTitle);
        const editorsChoiceReview = reviews.createEditorsChoiceReview(objectGraph, productData);
        const reviewSummary = reviews.createReviewSummaryProductReview(objectGraph, productData, shelfMetrics);
        const reviewSummaryForReviewsData = reviews.reviewSummaryFromData(objectGraph, productData, shelfMetrics);
        const allProductReviews = reviews.createProductReviewsList(objectGraph, ratingsData, reviewItems, true, editorsChoiceReview, reviewSummary, shelfMetrics);
        const userProductReviews = reviews.createProductReviewsList(objectGraph, ratingsData, reviewItems, true, null, null, shelfMetrics);
        const productReviewActions = createProductReviewActions(objectGraph, productData, shelfContext.productTitle, ratingsData, shelfContext.isBundle, tvOnlyApp, appIcon, lockupSubtitle);
        const includeMoreAction = objectGraph.client.isVision;
        const includeFeedbackActions = objectGraph.client.isVision;
        const seeAllAction = tvOnlyApp || reviewItems.length === 0 || (objectGraph.client.isVision && reviewsData.next == null)
            ? null
            : reviews.reviewsPageActionFromReviewsData(objectGraph, objectGraph.client.guid, ratingsData, reviewItems, reviewsData.next, shelfContext.productTitle, appIcon, includeMoreAction, includeFeedbackActions, productData, shelfMetrics);
        if (shelfContext.shouldShowRatingsAndReviews) {
            const ratingsShelf = new models.Shelf("productRatings");
            ratingsShelf.title = reviewsTitle;
            ratingsShelf.items = [productRatings];
            productReviewsShelves.ratingsShelf = ratingsShelf;
            shelfMetrics.addImpressionsToShelf(objectGraph, ratingsShelf, "ratingsOverview", null, null, null, objectGraph.loc.string("ProductPage.Section.Reviews.Title"));
            const allReviewActionsShelfItems = [];
            if (serverData.isDefinedNonNullNonEmpty(tapToRateProductReviewAction) && isReviewable) {
                allReviewActionsShelfItems.push(tapToRateProductReviewAction);
            }
            if (serverData.isDefinedNonNullNonEmpty(writeAReview) && isReviewable) {
                allReviewActionsShelfItems.push(writeAReview);
            }
            if (serverData.isDefinedNonNullNonEmpty(allReviewActionsShelfItems)) {
                const allReviewActionsShelf = new models.Shelf("productReviewAction");
                allReviewActionsShelf.items = allReviewActionsShelfItems;
                productReviewsShelves.allReviewActionsShelf = allReviewActionsShelf;
            }
            if (serverData.isDefinedNonNullNonEmpty(reviewSummaryForReviewsData)) {
                const reviewSummaryShelf = new models.Shelf("reviewSummary");
                reviewSummaryShelf.items = [reviewSummaryForReviewsData];
                productReviewsShelves.reviewSummaryShelf = reviewSummaryShelf;
            }
            let tapToRateProductReviewActionsShelf = null;
            if (serverData.isDefinedNonNullNonEmpty(tapToRateProductReviewAction) && isReviewable) {
                tapToRateProductReviewActionsShelf = new models.Shelf("productReviewAction");
                tapToRateProductReviewActionsShelf.items = [tapToRateProductReviewAction];
                productReviewsShelves.tapToRateActionsShelf = tapToRateProductReviewActionsShelf;
            }
            let writeAReviewActionsShelf = null;
            if (serverData.isDefinedNonNullNonEmpty(writeAReview) && isReviewable) {
                writeAReviewActionsShelf = new models.Shelf("productReviewAction");
                writeAReviewActionsShelf.items = [writeAReview];
                productReviewsShelves.writeAReviewActionsShelf = writeAReviewActionsShelf;
            }
            const userReviewsTitle = objectGraph.loc.string("ProductPage.Section.Reviews.MostHelpfulReviews.Title");
            const reviewSummaryTitle = objectGraph.loc.string("ProductPage.ReviewSummary.Body.Title");
            const hasUserReviews = isSome(userProductReviews) && userProductReviews.length > 0;
            let allProductReviewsShelf;
            if (isSome(allProductReviews) && allProductReviews.length > 0) {
                allProductReviewsShelf = new models.Shelf("productReview");
                allProductReviewsShelf.items = allProductReviews;
                allProductReviewsShelf.isHorizontal = true;
                if (objectGraph.client.isiOS) {
                    if (hasUserReviews) {
                        allProductReviewsShelf.title = userReviewsTitle;
                        allProductReviewsShelf.contentsMetadata = {
                            type: "productReviewsSection",
                            hasReviewSummary: isSome(reviewSummary),
                            reviewSummaryTitle: reviewSummaryTitle,
                        };
                        shelfMetrics.addImpressionsToShelf(objectGraph, allProductReviewsShelf, "mostHelpfulReviews", null, null, null, objectGraph.loc.string("ProductPage.Section.Reviews.MostHelpfulReviews.Title"));
                    }
                    else {
                        allProductReviewsShelf.title = reviewSummaryTitle;
                    }
                }
                productReviewsShelves.allProductReviewsShelf = allProductReviewsShelf;
            }
            let userProductReviewsShelf;
            if (hasUserReviews) {
                userProductReviewsShelf = new models.Shelf("productReview");
                userProductReviewsShelf.items = userProductReviews;
                userProductReviewsShelf.isHorizontal = true;
                if (objectGraph.client.isiOS) {
                    userProductReviewsShelf.title = userReviewsTitle;
                }
                shelfMetrics.addImpressionsToShelf(objectGraph, userProductReviewsShelf, "mostHelpfulReviews", null, null, null, objectGraph.loc.string("ProductPage.Section.Reviews.MostHelpfulReviews.Title"));
                productReviewsShelves.userProductReviewsShelf = userProductReviewsShelf;
            }
            if (objectGraph.client.deviceType === "tv") {
                const shelf = new models.Shelf("productRatingsAndReviewsComponent");
                shelf.title = reviewsTitle;
                const shelfItems = [];
                if (serverData.isDefinedNonNullNonEmpty(tapToRateProductReviewAction) &&
                    tapToRateProductReviewAction.action instanceof models.TapToRate &&
                    isReviewable) {
                    shelfItems.push(tapToRateProductReviewAction.action);
                }
                if (serverData.isDefinedNonNullNonEmpty(productRatings.status)) {
                    if (objectGraph.featureFlags.isEnabled("tv_product_page_2024E")) {
                        shelfItems.push(new models.ProductRatingsAndReviewsMessage(productRatings.status));
                    }
                    else {
                        shelfItems.push(reviews.noRatingsFromRatings(productRatings));
                    }
                }
                else {
                    shelfItems.push(reviews.starRatingsFromRatings(productRatings));
                    shelfItems.push(reviews.starRatingsHistogramFromRatings(productRatings));
                }
                shelf.items = shelfItems;
                productReviewsShelves.purchasedRatingsAndReviewsComponentShelf = shelf;
            }
            if (objectGraph.client.deviceType === "tv") {
                const shelf = new models.Shelf("productRatingsAndReviewsComponent");
                shelf.title = reviewsTitle;
                const shelfItems = [];
                if (serverData.isDefinedNonNullNonEmpty(productRatings.status)) {
                    if (objectGraph.featureFlags.isEnabled("tv_product_page_2024E")) {
                        shelfItems.push(new models.ProductRatingsAndReviewsMessage(productRatings.status));
                    }
                    else {
                        shelfItems.push(reviews.noRatingsFromRatings(productRatings));
                    }
                }
                else {
                    shelfItems.push(reviews.starRatingsFromRatings(productRatings));
                    shelfItems.push(reviews.starRatingsHistogramFromRatings(productRatings));
                }
                shelf.items = shelfItems;
                productReviewsShelves.notPurchasedRatingsAndReviewsComponentShelf = shelf;
            }
            if (objectGraph.client.isVision) {
                const productReviewsHeader = createProductReviewsLearnMoreShelf(objectGraph);
                if (isSome(productReviewsHeader)) {
                    productReviewsHeader.title = reviewsTitle;
                    productReviewsHeader.seeAllAction = seeAllAction;
                    productReviewsShelves.productReviewsHeader = productReviewsHeader;
                }
                // If there is a learn more shelf, the title and see all action is applied to the learn more shelf,
                // not the actual ratings and reviews shelf, due to the learn more text being placed directly
                // underneath the shelf title.
                const shelfTitle = isSome(productReviewsHeader) ? null : reviewsTitle;
                const shelfSeeAllAction = isSome(productReviewsHeader) ? null : seeAllAction;
                const purchasedShelf = createVisionRatingsAndReviewsShelf(objectGraph, true, shelfTitle, productRatings, productRatingsWereReset, productReviewActions, allProductReviews, shelfSeeAllAction);
                const notPurchasedShelf = createVisionRatingsAndReviewsShelf(objectGraph, false, shelfTitle, productRatings, productRatingsWereReset, productReviewActions, allProductReviews, seeAllAction);
                productReviewsShelves.purchasedRatingsAndReviewsComponentShelf = purchasedShelf;
                productReviewsShelves.notPurchasedRatingsAndReviewsComponentShelf = notPurchasedShelf;
            }
            let editorsChoiceProductReviewsShelf = null;
            if (serverData.isDefinedNonNullNonEmpty(editorsChoiceReview)) {
                editorsChoiceProductReviewsShelf = new models.Shelf("productReview");
                editorsChoiceProductReviewsShelf.items = [editorsChoiceReview];
                if (!shelfContext.shouldShowRatingsAndReviews) {
                    editorsChoiceProductReviewsShelf.title = reviewsTitle;
                }
                else if (objectGraph.client.isiOS &&
                    !editorsChoiceReview.review.showsBadge) {
                    editorsChoiceProductReviewsShelf.title = objectGraph.loc.string("ProductPage.Section.EditorsNotes.Title");
                }
                productReviewsShelves.editorsChoiceProductReviewsShelf = editorsChoiceProductReviewsShelf;
            }
            // Web does not currently support a "See All" page for Bundles
            const isBundleOnWeb = shelfContext.isBundle && objectGraph.client.isWeb;
            if (!tvOnlyApp && !isBundleOnWeb && reviewItems.length > 0) {
                productReviewsShelves.ratingsShelf.seeAllAction = objectGraph.client.isiOS
                    ? createReviewsPageSeeAllAction(objectGraph, objectGraph.client.guid, productData.id, ratingsData, reviewItems, reviewsData.next, includeMoreAction, includeFeedbackActions, productReviewsShelves)
                    : reviews.reviewsPageActionFromReviewsData(objectGraph, objectGraph.client.guid, ratingsData, reviewItems, reviewsData.next, shelfContext.productTitle, appIcon, includeMoreAction, includeFeedbackActions, productData, shelfMetrics);
            }
        }
        return productReviewsShelves;
    });
}
function createProductRatings(objectGraph, ratingsData, reviewsCount, wereReset) {
    return validation.context("createProductRatings", () => {
        const ratings = reviews.ratingsFromApiResponses(objectGraph, objectGraph.client.guid, "productPage", ratingsData);
        const hasRatings = ratings.ratingAverage > 0 && ratings.ratingCounts;
        if (!hasRatings && !wereReset) {
            ratings.status = objectGraph.loc.string("RATINGS_STATUS_NOT_ENOUGH_RATINGS_OR_REVIEWS");
        }
        return ratings;
    });
}
function createTapToRateProductReviewAction(objectGraph, productData, productTitle, ratingsData, isBundle, tvOnlyApp) {
    return validation.context("createTapToRateProductReviewAction", () => {
        const tapToRate = createTapToRate(objectGraph, productData, productTitle, ratingsData, isBundle, tvOnlyApp);
        if (tapToRate) {
            const productReviewAction = new models.ProductReviewAction();
            productReviewAction.actionType = "tapToRate";
            productReviewAction.action = tapToRate;
            productReviewAction.id = "ProductReviewAction.TapToRate";
            return productReviewAction;
        }
        else {
            return null;
        }
    });
}
/**
 * Creates the TapToRate object, for tapping on a star to rate the app
 * @param objectGraph Current object graph
 * @param productData The data blob for the product
 * @param productTitle The title of the product
 * @param ratingsData The data blobs for the ratings
 * @param isBundle Whether this product is for a bundle
 * @param tvOnlyApp Whether this product is for a TV only app
 * @returns The built TapToRate object
 */
function createTapToRate(objectGraph, productData, productTitle, ratingsData, isBundle, tvOnlyApp) {
    return validation.context("createTapToRateProductReviewAction", () => {
        // Tap to Rate
        if (!tvOnlyApp || objectGraph.client.isTV) {
            return reviews.tapToRateWithAdamId(objectGraph, productData.id, isBundle, productTitle, null);
        }
        else {
            return null;
        }
    });
}
/**
 * Creates an action for writing a product review
 * @param objectGraph Current object graph
 * @param productData The data blob for the product
 * @param ratingsData The data blobs for the ratings
 * @param tvOnlyApp Whether this product is for a TV only app
 * @param appIcon The icon artwork for the product
 * @returns A product review action
 */
function createWriteAReview(objectGraph, productData, ratingsData, tvOnlyApp, appIcon, lockupSubtitle, lockupTitle) {
    return validation.context("createWriteAReview", () => {
        // Tap to Rate
        if (!tvOnlyApp) {
            const productWriteAReview = new models.ProductWriteAReview();
            productWriteAReview.writeReviewAction = reviews.writeReviewActionFromData(objectGraph, productData, lockupSubtitle, lockupTitle, appIcon);
            productWriteAReview.supportAction = createSupportAction(objectGraph, ratingsData);
            const productReviewAction = new models.ProductReviewAction();
            productReviewAction.actionType = "writeAReview";
            productReviewAction.action = productWriteAReview;
            return productReviewAction;
        }
        else {
            return null;
        }
    });
}
/**
 * Creates an action for contacting support
 * @param objectGraph Current object graph
 * @param ratingsData The data blobs for the ratings
 * @returns An external URL action
 */
function createSupportAction(objectGraph, ratingsData) {
    const supportUrl = serverData.asString(ratingsData, "supportUrl");
    if (supportUrl) {
        const supportAction = new models.ExternalUrlAction(supportUrl, false);
        supportAction.title = objectGraph.loc.string("APP_SUPPORT");
        supportAction.artwork = createArtworkForResource(objectGraph, "systemimage://questionmark.circle");
        return supportAction;
    }
    else {
        return null;
    }
}
/**
 * Creates a ProductReviewActions object, which includes tap to rate, write a review, and support
 * @param objectGraph Current object graph
 * @param productData The data blob for the product
 * @param productTitle The title of the product
 * @param ratingsData The data blobs for the ratings
 * @param isBundle Whether this product is for a bundle
 * @param tvOnlyApp Whether this product is for a TV only app
 * @param appIcon The icon artwork for the product
 * @returns The built ReviewActions object
 */
export function createProductReviewActions(objectGraph, productData, productTitle, ratingsData, isBundle, tvOnlyApp, appIcon, lockupSubtitle) {
    return validation.context("createProductReviewActions", () => {
        const productReviewActions = new models.ProductReviewActions();
        productReviewActions.tapToRate = createTapToRate(objectGraph, productData, productTitle, ratingsData, isBundle, tvOnlyApp);
        productReviewActions.writeReviewAction = reviews.writeReviewActionFromData(objectGraph, productData, lockupSubtitle, productTitle, appIcon);
        productReviewActions.supportAction = createSupportAction(objectGraph, ratingsData);
        return productReviewActions;
    });
}
/**
 * Creates a ratings and reviews shelf for visionOS, which is composed of a
 * number of different ProductRatingsAndReviewsComponent items
 * @param objectGraph Current object graph
 * @param isPurchasedShelf Whether this shelf is for display for a purchased app
 * @param shelfTitle The title to use for the shelf
 * @param productRatings The ratings for the product
 * @param productRatingsWereReset Whether the product ratings were reset by the developer
 * @param reviewActions A review actions object
 * @param productReviews An array of product reviews
 * @param seeAllAction The see all action for the shelf
 * @returns The built shelf
 */
function createVisionRatingsAndReviewsShelf(objectGraph, isPurchasedShelf, shelfTitle, productRatings, productRatingsWereReset, productReviewActions, productReviews, seeAllAction) {
    const shelf = new models.Shelf("productRatingsAndReviewsComponent");
    shelf.title = shelfTitle;
    let shelfItems = [];
    const hasRatings = productRatings.ratingAverage > 0 && productRatings.ratingCounts;
    const hasReviews = productReviews.length > 0;
    if (hasRatings) {
        shelfItems.push(reviews.starRatingsFromRatings(productRatings));
    }
    else if (productRatingsWereReset) {
        const messageText = objectGraph.loc.string("RATINGS_STATUS_DEVELOPER_RESET");
        shelfItems.push(new models.ProductRatingsAndReviewsMessage(messageText));
    }
    else if (hasReviews) {
        const messageText = objectGraph.loc.string("RATINGS_STATUS_NOT_ENOUGH_RATINGS");
        shelfItems.push(new models.ProductRatingsAndReviewsMessage(messageText));
    }
    if (isPurchasedShelf && serverData.isDefinedNonNullNonEmpty(productReviewActions)) {
        shelfItems.push(productReviewActions);
    }
    if (hasReviews) {
        shelfItems = shelfItems.concat(productReviews);
    }
    else if (!hasRatings && !productRatingsWereReset) {
        const messageText = objectGraph.loc.string("RATINGS_STATUS_NOT_ENOUGH_RATINGS_OR_REVIEWS");
        shelfItems.push(new models.ProductRatingsAndReviewsMessage(messageText));
    }
    else {
        const messageText = objectGraph.loc.string("RATINGS_STATUS_NO_REVIEWS");
        shelfItems.push(new models.ProductRatingsAndReviewsMessage(messageText));
    }
    shelf.items = shelfItems;
    shelf.isHorizontal = true;
    shelf.seeAllAction = seeAllAction;
    if (shelfItems.filter((shelfItem) => shelfItem instanceof models.ProductRatingsAndReviewsMessage).length >= 1) {
        if (shelfItems.length === 1) {
            shelf.presentationHints = { isSingleDensity: true };
        }
        else if (shelfItems.length === 2) {
            shelf.presentationHints = { isLowDensity: true };
        }
    }
    return shelf;
}
/**
 * Creates a flow action to display a reviews page.
 * @param deviceId  The UUID for the user's device.
 * @param adamId The adamid of the app
 * @param ratingsData The ratings data response
 * @param reviewsData The review row data response.
 * @param nextPageHref The Media API href for the next page.
 * @param appName The name of the app
 * @param appIcon The app's icon.
 * @param productData The data for the product

 * @returns A `FlowAction` object pointing to a reviews page.
 */
function createReviewsPageSeeAllAction(objectGraph, deviceId, adamId, ratingsData, reviewItems, nextPageHref = null, includeMoreAction, includeFeedbackActions, reviewsShelves) {
    if (!ratingsData) {
        return null;
    }
    return validation.context("reviewsPageActionFromReviewsData", () => {
        var _a;
        const pageData = new models.ReviewsPage();
        let reviewActionsShelf;
        if (isSome(reviewsShelves === null || reviewsShelves === void 0 ? void 0 : reviewsShelves.allReviewActionsShelf)) {
            reviewActionsShelf = shallowCopyOf(reviewsShelves.allReviewActionsShelf);
            const existingReviewActionsShelfPresentationHints = (_a = reviewActionsShelf.presentationHints) !== null && _a !== void 0 ? _a : {};
            reviewActionsShelf.presentationHints = {
                ...existingReviewActionsShelfPresentationHints,
                displayIfReviewable: true,
            };
            addSeeAllContextPresentationHint(reviewActionsShelf);
        }
        let ratingsShelf;
        if (isSome(reviewsShelves === null || reviewsShelves === void 0 ? void 0 : reviewsShelves.ratingsShelf)) {
            ratingsShelf = shallowCopyOf(reviewsShelves.ratingsShelf);
            const ratings = ratingsShelf.items[0];
            if (isSome(ratings) && ratingsShelf.items.length === 1) {
                const ratingsCopy = shallowCopyOf(ratings);
                ratingsCopy.context = "details";
                ratingsShelf.items = [ratingsCopy];
            }
            ratingsShelf.seeAllAction = null;
            ratingsShelf.title = null;
            addSeeAllContextPresentationHint(ratingsShelf);
        }
        let reviewSummaryShelf;
        if (isSome(reviewsShelves === null || reviewsShelves === void 0 ? void 0 : reviewsShelves.reviewSummaryShelf)) {
            reviewSummaryShelf = shallowCopyOf(reviewsShelves.reviewSummaryShelf);
            addSeeAllContextPresentationHint(reviewSummaryShelf);
        }
        const pageComponents = reviews.createReviewsPageComponents(objectGraph, deviceId, adamId, ratingsData, reviewItems, nextPageHref, includeMoreAction, includeFeedbackActions, "helpful", {
            ratingsShelf: ratingsShelf,
            reviewSummaryShelf: reviewSummaryShelf,
            reviewActionsShelf: reviewActionsShelf,
        });
        pageData.title = reviews.pageTitleForPlatform(objectGraph);
        pageData.adamId = serverData.asString(ratingsData, "adamId");
        pageData.nextPage = pageComponents.paginationToken;
        pageData.initialSortOptionIdentifier = pageComponents.initialSortId;
        pageData.sortActionSheetTitle = pageComponents.sortActionSheetTitle;
        pageData.sortOptions = pageComponents.sorts;
        if (isSome(ratingsShelf)) {
            pageData.shelves.push(ratingsShelf);
        }
        if (isSome(reviewSummaryShelf)) {
            pageData.shelves.push(reviewSummaryShelf);
        }
        if (isSome(reviewActionsShelf)) {
            pageData.shelves.push(reviewActionsShelf);
        }
        pageData.shelves.push(pageComponents.reviewsShelf);
        const reviewSummaryLearnMoreAction = reviews.reviewSummaryLearnMoreAction(objectGraph);
        if (isSome(reviewSummaryLearnMoreAction)) {
            pageData.trailingNavBarAction = reviewSummaryLearnMoreAction;
        }
        const action = new models.FlowAction("reviews");
        action.pageData = pageData;
        action.title = objectGraph.loc.string("ACTION_SEE_ALL");
        return action;
    });
}
/**
 * Sets `isSeeAllContext` to `true` in the presentation hints of the provided shelf.
 * @param shelf The shelf to add the presentation hint to
 */
function addSeeAllContextPresentationHint(shelf) {
    var _a;
    if (isSome(shelf)) {
        const existingPresentationHints = (_a = shelf.presentationHints) !== null && _a !== void 0 ? _a : {};
        shelf.presentationHints = {
            ...existingPresentationHints,
            isSeeAllContext: true,
        };
    }
}
/**
 * Creates a "Learn More"  shelf for product reviews, which includes transparency info about the feature.
 * This can be placed in different locations on a page based on the platform.
 * @param objectGraph Current object graph
 * @returns The built shelf
 */
export function createProductReviewsLearnMoreShelf(objectGraph) {
    if (serverData.isNullOrEmpty(objectGraph.bag.ratingsAndReviewsLearnMoreEditorialId)) {
        return null;
    }
    if (!objectGraph.client.isiOS && !objectGraph.client.isVision) {
        return null;
    }
    const shelf = new models.Shelf("linkableText");
    const bodyText = bodyTextForProductReviewsLearnMore(objectGraph);
    if (isNothing(bodyText)) {
        return null;
    }
    shelf.items = [bodyText];
    return shelf;
}
/**
 * Creates the linkable body text for the product reviews learn more shelf.
 * @param objectGraph Current object graph
 * @returns The built body text
 */
function bodyTextForProductReviewsLearnMore(objectGraph) {
    const learnMoreLinkText = objectGraph.loc.string("ProductPage.RatingsAndReviews.Footer.LearnMore");
    const linkedSubstrings = {};
    const learnMoreAction = createProductReviewsLearnMoreAction(objectGraph);
    if (isNothing(learnMoreAction)) {
        return null;
    }
    linkedSubstrings[learnMoreLinkText] = learnMoreAction;
    let text = objectGraph.loc.string("ProductPage.RatingsAndReviews.Footer");
    text = text.replace("{learnMoreLink}", learnMoreLinkText);
    const textType = "text/plain";
    const styledText = new models.StyledText(text, textType);
    return new models.LinkableText(styledText, linkedSubstrings);
}
/**
 * Creates the FlowAction to navigate to the article related to the product reviews learn more shelf.
 * @param objectGraph Current object graph
 * @returns The built FlowAction
 */
function createProductReviewsLearnMoreAction(objectGraph) {
    const editorialItemId = objectGraph.bag.ratingsAndReviewsLearnMoreEditorialId;
    if (serverData.isNullOrEmpty(editorialItemId)) {
        return null;
    }
    const learnMoreAction = new models.FlowAction("article");
    learnMoreAction.title = objectGraph.loc.string("ProductPage.RatingsAndReviews.Footer.LearnMore");
    learnMoreAction.pageUrl = `https://apps.apple.com/story/id${editorialItemId}`;
    return learnMoreAction;
}
/**
 * Creates the Review Summary Shelf
 * @param objectGraph Current object graph
 * @param data The current product page data
 * @returns The review summary shelf
 */
export function createReviewSummary(objectGraph, data, shelfMetrics) {
    const reviewSummary = reviews.reviewSummaryFromData(objectGraph, data, shelfMetrics);
    if (isNothing(reviewSummary)) {
        return null;
    }
    const shelf = new models.Shelf("reviewSummary");
    shelf.items = [reviewSummary];
    return shelf;
}
//# sourceMappingURL=shelf-based-reviews-shelves.js.map