import { isSome } from "@jet/environment";
import * as validation from "@jet/environment/json/validation";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as mediaRelationship from "../../../foundation/media/relationships";
import * as content from "../../content/content";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as metricsHelpersLocation from "../../metrics/helpers/location";
import { insertPlaceholdersIntoProductPageShelfIfRequired } from "../../placeholders/placeholders";
import * as productPageCommon from "../product-page-common";
import { ProductPageOnDemandShelfType } from "../product-page-common";
import { ProductPageShelfMetrics } from "../product-page-shelf-metrics";
import { getLocale } from "../../locale";
import { makeDeveloperPageIntent } from "../../../api/intents/developer-page-intent";
/**
 * Create a shelf of developers other apps for the product page.
 *
 * @param productData The raw data response for a product page JSON fetch.
 * @param shelfMetrics The product page shelf metrics.
 * @returns A `More by developer` shelf.
 */
export function createInitialShelf(objectGraph, productData, shelfMetrics) {
    return validation.context("moreByDeveloperShelf", () => {
        if (serverData.isNullOrEmpty(productData)) {
            return null;
        }
        const developer = mediaRelationship.relationshipData(objectGraph, productData, "developer");
        const developerUrl = content.developerUrlFromDeveloperData(objectGraph, developer);
        const developerName = mediaAttributes.attributeAsString(productData, "artistName");
        let otherAppsRelationship;
        let otherApps;
        if (preprocessor.GAMES_TARGET) {
            otherApps = mediaRelationship.relationshipViewsCollection(productData, "developer-other-games");
            if (otherApps.length === 0) {
                return null;
            }
        }
        else {
            otherAppsRelationship = mediaRelationship.relationship(productData, "developer-other-apps");
            if (!otherAppsRelationship || otherAppsRelationship.data.length === 0) {
                return null;
            }
            otherApps = otherAppsRelationship.data;
        }
        const title = objectGraph.loc
            .string("ProductPage.Section.MoreByDeveloper.TitleTemplate")
            .replace("{developer}", developerName);
        metricsHelpersLocation.pushContentLocation(objectGraph, {
            pageInformation: shelfMetrics.metricsPageInformation,
            locationTracker: shelfMetrics.locationTracker,
            targetType: "swoosh",
            id: `${shelfMetrics.getSequenceId()}`,
            idType: "sequential",
        }, title);
        let contentType;
        switch (objectGraph.client.deviceType) {
            case "mac":
            case "tv":
                contentType = "mediumLockup";
                break;
            default:
                contentType = "smallLockup";
                break;
        }
        const shelf = new models.Shelf(contentType);
        shelf.title = title;
        const hasShelfBackground = !objectGraph.client.isTV && !objectGraph.client.isiOS;
        const offerStyle = hasShelfBackground ? "white" : null;
        const filter = objectGraph.client.isCompanionVisionApp ? 32768 /* Filter.VisionOSCompatibility */ : undefined;
        let shelfItems;
        if (isSome(otherAppsRelationship)) {
            shelfItems = productPageCommon.lockupsFromDataContainer(objectGraph, otherAppsRelationship, shelfMetrics, content.artworkUseCaseFromShelfStyle(objectGraph, contentType), contentType, offerStyle, filter);
        }
        else if (isSome(otherApps)) {
            shelfItems = productPageCommon.lockupsFromData(objectGraph, otherApps, shelfMetrics, content.artworkUseCaseFromShelfStyle(objectGraph, contentType), contentType, offerStyle, filter);
        }
        if (!shelfItems) {
            return null;
        }
        shelf.items = shelfItems.items;
        shelf.isHorizontal = true;
        if (otherApps.length < 2 || objectGraph.client.isTV) {
            shelf.rowsPerColumn = 1;
        }
        else {
            shelf.rowsPerColumn = 2;
        }
        if (!objectGraph.client.isTV) {
            const developerSeeAll = new models.FlowAction("page");
            developerSeeAll.pageUrl = developerUrl;
            developerSeeAll.title = objectGraph.loc.string("ACTION_SEE_ALL");
            metricsHelpersClicks.addClickEventToSeeAllAction(objectGraph, developerSeeAll, developerSeeAll.pageUrl, {
                pageInformation: shelfMetrics.metricsPageInformation,
                locationTracker: shelfMetrics.locationTracker,
            });
            if (objectGraph.client.isWeb) {
                developerSeeAll.destination = makeDeveloperPageIntent({
                    ...getLocale(objectGraph),
                    id: developer.id,
                });
            }
            shelf.seeAllAction = developerSeeAll;
        }
        if (hasShelfBackground) {
            shelf.background = {
                type: "color",
                color: productPageCommon.grayShelfBackgroundColor,
            };
        }
        if (objectGraph.client.isiOS && objectGraph.bag.isOnDemandShelfFetchingEnabled) {
            shelf.fetchStrategy = models.IncompleteShelfFetchStrategy.OnShelfWillAppear;
        }
        if (serverData.isDefinedNonNullNonEmpty(shelfItems.remainingItems)) {
            metricsHelpersLocation.popLocation(shelfMetrics.locationTracker);
            shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "developer");
            metricsHelpersLocation.nextPosition(shelfMetrics.locationTracker);
            const shelfToken = new productPageCommon.ProductPageShelfToken(productData.id, shelfItems.remainingItems, shelf.title, false, undefined, contentType, offerStyle, null, null, null, null, ProductPageOnDemandShelfType.MoreByDeveloper);
            shelfToken.developerUrl = developerUrl;
            insertPlaceholdersIntoProductPageShelfIfRequired(objectGraph, shelf, shelfToken);
            shelf.url = productPageCommon.shelfContentUrl(objectGraph, shelfToken, shelfMetrics);
        }
        else {
            metricsHelpersLocation.popLocation(shelfMetrics.locationTracker);
            shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "developer");
            metricsHelpersLocation.nextPosition(shelfMetrics.locationTracker);
        }
        return shelf;
    });
}
/**
 * Create a shelf of more by developer items for the secondary look up.
 *
 * @param dataItems The raw data items from the secondary fetch.
 * @param shelfToken The shelf token for this shelf.
 * @returns A more by developer items shelf.
 */
export function createSecondaryShelf(objectGraph, dataItems, shelfToken) {
    return validation.context("moreByDeveloperSecondaryShelf", () => {
        if (serverData.isNullOrEmpty(dataItems)) {
            return null;
        }
        const shelfMetrics = new ProductPageShelfMetrics(shelfToken.sourcePageInformation, shelfToken.sourceLocationTracker, shelfToken.sourceSequenceId);
        const metricsOptions = {
            pageInformation: shelfMetrics.metricsPageInformation,
            locationTracker: shelfMetrics.locationTracker,
            targetType: "swoosh",
            id: `${shelfMetrics.getSequenceId()}`,
            idType: "sequential",
            excludeAttribution: true,
            recoMetricsData: shelfToken.recoMetricsData,
        };
        metricsHelpersLocation.pushContentLocation(objectGraph, metricsOptions, shelfToken.title);
        const shelf = new models.Shelf(shelfToken.contentType);
        shelf.title = shelfToken.title;
        const hasShelfBackground = !objectGraph.client.isTV && !objectGraph.client.isiOS;
        const filter = objectGraph.client.isCompanionVisionApp ? 32768 /* Filter.VisionOSCompatibility */ : undefined;
        const shelfItems = productPageCommon.lockupsFromData(objectGraph, dataItems, shelfMetrics, content.artworkUseCaseFromShelfStyle(objectGraph, shelfToken.contentType), shelfToken.contentType, shelfToken.offerStyle, filter);
        if (serverData.isNullOrEmpty(shelfItems)) {
            return null;
        }
        shelf.items = shelfItems.items;
        shelf.isHorizontal = true;
        if (dataItems.length < 2 || objectGraph.client.isTV) {
            shelf.rowsPerColumn = 1;
        }
        else {
            shelf.rowsPerColumn = 2;
        }
        if (!objectGraph.client.isTV && isSome(shelfToken.developerUrl)) {
            const developerSeeAll = new models.FlowAction("page");
            developerSeeAll.pageUrl = shelfToken.developerUrl;
            developerSeeAll.title = objectGraph.loc.string("ACTION_SEE_ALL");
            metricsHelpersClicks.addClickEventToSeeAllAction(objectGraph, developerSeeAll, developerSeeAll.pageUrl, {
                pageInformation: shelfMetrics.metricsPageInformation,
                locationTracker: shelfMetrics.locationTracker,
            });
            shelf.seeAllAction = developerSeeAll;
        }
        if (hasShelfBackground) {
            shelf.background = {
                type: "color",
                color: productPageCommon.grayShelfBackgroundColor,
            };
        }
        shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "developer");
        metricsHelpersLocation.popLocation(shelfMetrics.locationTracker);
        metricsHelpersLocation.nextPosition(shelfMetrics.locationTracker);
        return shelf;
    });
}
//# sourceMappingURL=more-by-developer-shelf.js.map