import * as validation from "@jet/environment/json/validation";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import { Parameters, Path, Protocol } from "../../../foundation/network/url-constants";
import * as contentAttributes from "../../content/attributes";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as metricsHelpersImpressions from "../../metrics/helpers/impressions";
import * as metricsHelpersLocation from "../../metrics/helpers/location";
import { platformPrefersLargeTitles } from "../../room/room-common";
import * as productPageCommon from "../product-page-common";
import { makeGameCenterHeader, openGamesUIAction } from "../../arcade/arcade-common";
/**
 * Create a shelf of Game Center friends who play this game.
 *
 * @param productData The raw data response for a product page JSON fetch.
 * @param shelfMetrics The product page shelf metrics.
 * @param pageInformation
 * @returns A `Friends who played this game` shelf.
 */
export function create(objectGraph, productData, shelfMetrics, shelfContext) {
    return validation.context("friendsPlayingShelf", () => {
        if (serverData.isNullOrEmpty(productData)) {
            return null;
        }
        const adamId = productData.id;
        if (serverData.isNullOrEmpty(adamId)) {
            return null;
        }
        if (objectGraph.client.isWeb) {
            return null;
        }
        const isGameCenterEnabled = contentAttributes.contentAttributeAsBooleanOrFalse(objectGraph, productData, "isGameCenterEnabled");
        if (!isGameCenterEnabled) {
            return null;
        }
        // rdar://70124826 (N104/18C23a: friends shown as playing a game when it\u2019s not available yet)
        if (shelfContext.isPreorder) {
            return null;
        }
        const title = objectGraph.loc.string("ProductPage.Section.FriendsPlaying.Title", "Playing This Game");
        const metricsOptions = {
            pageInformation: shelfMetrics.metricsPageInformation,
            locationTracker: shelfMetrics.locationTracker,
            targetType: "swoosh",
            id: `${shelfMetrics.getSequenceId()}`,
            idType: "sequential",
        };
        metricsHelpersLocation.pushContentLocation(objectGraph, metricsOptions, title);
        const shelf = new models.Shelf("largeGameCenterPlayer");
        shelf.header = makeGameCenterHeader(objectGraph, title);
        shelf.isHidden = true;
        shelf.isHorizontal = true;
        // This shelf is always empty initially; it is populated natively and then updated in JS using the function `createShelfWithFriends`.
        shelf.items = [];
        shelf.mergeWhenFetched = true;
        const token = new productPageCommon.ProductPageShelfToken(productData.id, [], shelf.title, false, undefined, null, null, null);
        shelf.url = `${Protocol.internal}:/${Path.product}/${Path.shelf}/?${Parameters.isGameCenterPlayerShelf}=true&${Parameters.id}=${adamId}&${Parameters.token}=${productPageCommon.encodedShelfToken(token, shelfMetrics, shelfMetrics.metricsPageInformation)}`;
        shelf.batchGroup = "gameCenter";
        shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "friendsPlaying");
        metricsHelpersLocation.popLocation(shelfMetrics.locationTracker);
        metricsHelpersLocation.nextPosition(shelfMetrics.locationTracker);
        const seeAllAction = new models.FlowAction("page");
        metricsHelpersClicks.addClickEventToSeeAllAction(objectGraph, seeAllAction, null, {
            pageInformation: shelfMetrics.metricsPageInformation,
            locationTracker: shelfMetrics.locationTracker,
        });
        return shelf;
    });
}
export function createShelfWithFriends(objectGraph, gameCenterPlayers, overflowToSeeAll, token) {
    const maxNumberOfPlayersBeforeSeeAll = objectGraph.client.isiOS ? 8 : 20;
    const friendPrefix = "FRIEND";
    const shelf = new models.Shelf("largeGameCenterPlayer");
    shelf.isHorizontal = true;
    let visiblePlayers = null;
    if (overflowToSeeAll) {
        visiblePlayers = gameCenterPlayers.slice(0, maxNumberOfPlayersBeforeSeeAll);
    }
    else {
        visiblePlayers = gameCenterPlayers;
    }
    const shelfItems = [];
    for (let index = 0; index < visiblePlayers.length; index++) {
        const player = visiblePlayers[index];
        player.action = objectGraph.client.isTV
            ? new models.GameCenterPlayerProfileAction(player.playerId)
            : new models.OpenGamesUIAction({ player: { playerID: player.playerId } });
        if (serverData.isDefinedNonNullNonEmpty(token)) {
            // Content metrics
            const contentKind = "friendPlayer";
            const contentMetricsOptions = {
                pageInformation: token.destinationPageInformation,
                locationTracker: token.sourceLocationTracker,
                id: "",
                idType: "sequential",
                anonymizationOptions: {
                    anonymizationString: `${friendPrefix}${index + 1}`,
                },
            };
            // Impression metrics
            const impressionMetricsOptions = {
                ...contentMetricsOptions,
                kind: contentKind,
                title: "",
                softwareType: null,
            };
            metricsHelpersImpressions.addImpressionFields(objectGraph, player, impressionMetricsOptions);
            if (objectGraph.bag.productPageFriendsPlayingClickEventsEnabled) {
                // Click metrics
                const clickOptions = {
                    ...contentMetricsOptions,
                    kind: contentKind,
                    targetType: "lockup",
                    actionType: "navigate",
                    softwareType: null,
                };
                metricsHelpersClicks.addClickEventToAction(objectGraph, player.action, clickOptions);
            }
        }
        shelfItems.push(player);
        if (serverData.isDefinedNonNullNonEmpty(token)) {
            metricsHelpersLocation.nextPosition(token.sourceLocationTracker);
        }
    }
    shelf.items = shelfItems;
    shelf.mergeWhenFetched = true;
    shelf.isHidden = shelf.items.length === 0;
    shelf.batchGroup = "gameCenter";
    if (overflowToSeeAll && gameCenterPlayers.length > maxNumberOfPlayersBeforeSeeAll) {
        // The shelf for the see all page
        const shelfForAllPlayers = new models.Shelf("smallGameCenterPlayer");
        shelfForAllPlayers.title = objectGraph.loc.stringWithCount("ProductPage.FriendsPlaying.SeeAllTitle", gameCenterPlayers.length);
        shelfForAllPlayers.items = gameCenterPlayers.map((player) => {
            player.action = new models.GameCenterPlayerProfileAction(player.playerId);
            return player;
        });
        shelfForAllPlayers.rowsPerColumn = 1;
        // See all page
        const seeAllPage = new models.GenericPage([shelfForAllPlayers]);
        seeAllPage.title = token === null || token === void 0 ? void 0 : token.title;
        if (platformPrefersLargeTitles(objectGraph)) {
            seeAllPage.presentationOptions = ["prefersLargeTitle"];
        }
        // The action that opens the see all page
        const seeAllAction = new models.FlowAction("page");
        seeAllAction.title = objectGraph.loc.string("ACTION_SEE_ALL");
        seeAllAction.pageData = seeAllPage;
        // Connect the shelf's seeAllAction
        shelf.seeAllAction = seeAllAction;
    }
    if (serverData.isDefinedNonNullNonEmpty(token)) {
        const shelfMetricsOptions = {
            id: "",
            kind: null,
            softwareType: null,
            targetType: "swoosh",
            title: token.title,
            pageInformation: token.sourcePageInformation,
            locationTracker: token.sourceLocationTracker,
            idType: "sequential",
            badges: {
                gameCenter: true,
            },
        };
        metricsHelpersImpressions.addImpressionFields(objectGraph, shelf, shelfMetricsOptions);
    }
    shelf.footerTitle = objectGraph.loc.string("Lockup.Footer.GamesApp");
    shelf.footerAction = openGamesUIAction(objectGraph, { friends: {} });
    shelf.footerStyle = {
        $kind: "games",
        bundleID: "com.apple.games",
        width: 16,
        height: 16,
    };
    return shelf;
}
//# sourceMappingURL=friends-playing-shelf.js.map