import * as validation from "@jet/environment/json/validation";
import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import { attributeAsString } from "@apple-media-services/media-api";
import { hrefToRoutableUrl } from "../../builders/url-mapping";
import { isNothing, isSome } from "@jet/environment";
import * as mediaRelationship from "../../../foundation/media/relationships";
import { pageRouter } from "../../builders/routing";
import * as contentAttributes from "../../content/attributes";
import * as content from "../../content/content";
import * as productPageVariants from "../product-page-variants";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as metricsHelpersImpressions from "../../metrics/helpers/impressions";
import * as metricsHelpersLocation from "../../metrics/helpers/location";
/**
 * Create a shelf for the product page description.
 *
 * @param data The raw data response for a product page JSON fetch.
 * @param shelfMetrics The product page shelf metrics.
 * @param shelfContext A collection of any other variables used when creating this shelf.
 * @returns A description shelf.
 */
export function create(objectGraph, data, shelfMetrics, shelfContext) {
    return validation.context("descriptionShelf", () => {
        if (serverData.isNullOrEmpty(data)) {
            return null;
        }
        const shelf = new models.Shelf("productDescription");
        const links = linksFromData(objectGraph, data);
        const tagRibbonItems = createTagsRibbonItems(objectGraph, data, shelfMetrics.metricsPageInformation, shelfMetrics.locationTracker);
        const paragraph = paragraphFromData(objectGraph, data);
        if (paragraph === null) {
            return null;
        }
        const productDescription = new models.ProductDescription(paragraph, links, tagRibbonItems, shelfContext.developerAction);
        shelf.items = [productDescription];
        if (objectGraph.client.isWatch) {
            shelf.title = objectGraph.loc.string("PRODUCT_DESCRIPTION");
        }
        shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "description");
        return shelf;
    });
}
export function createTagsRibbonItems(objectGraph, data, metricsPageInformation, metricsLocationTracker) {
    const tagsData = mediaRelationship.relationshipViewsCollection(data, "categorizations");
    const ribbonItems = [];
    for (const tagData of tagsData) {
        const name = attributeAsString(tagData, "name");
        if (isSome(name)) {
            let pageUrl = null;
            switch (tagData.type) {
                case "genres":
                    pageUrl = attributeAsString(tagData, "url");
                    break;
                case "tags":
                    const href = serverData.asString(tagData, "href");
                    if (href) {
                        pageUrl = hrefToRoutableUrl(objectGraph, href);
                    }
                    break;
                default:
                    break;
            }
            if (isNothing(pageUrl)) {
                continue;
            }
            const flowPage = objectGraph.required(pageRouter).fetchFlowPage(pageUrl);
            const flowAction = new models.FlowAction(flowPage);
            flowAction.pageUrl = pageUrl;
            flowAction.title = name;
            const ribbonItem = new models.RibbonBarItem(name, flowAction);
            // Configure impressions
            const metricsOptions = {
                targetType: "facet",
                pageInformation: metricsPageInformation,
                locationTracker: metricsLocationTracker,
                recoMetricsData: null,
            };
            const impressionOptions = metricsHelpersImpressions.impressionOptionsForTagRibbonItem(objectGraph, tagData, name, metricsOptions);
            metricsHelpersImpressions.addImpressionFields(objectGraph, ribbonItem, impressionOptions);
            const metricsClickOptions = metricsHelpersClicks.clickOptionsForLockup(objectGraph, data, metricsOptions);
            metricsClickOptions.targetType = metricsOptions.targetType;
            metricsHelpersClicks.addClickEventToAction(objectGraph, flowAction, metricsClickOptions);
            ribbonItems.push(ribbonItem);
            metricsHelpersLocation.nextPosition(metricsOptions.locationTracker);
        }
    }
    return ribbonItems;
}
/**
 * Create a description for the product.
 *
 * @param data The raw data response for a product page JSON fetch.
 * @returns The raw description as a paragraph.
 */
export function paragraphFromData(objectGraph, data, productVariantData = null) {
    return validation.context("descriptionFromData", () => {
        if (serverData.isNull(productVariantData)) {
            productVariantData = productPageVariants.productVariantDataForData(objectGraph, data); // create variant data if not provided by caller.
        }
        let promotionalText = content.promotionalTextFromData(objectGraph, data, productVariantData);
        if (!promotionalText) {
            promotionalText = "";
        }
        let mainDescription = contentAttributes.contentAttributeAsString(objectGraph, data, "description.standard");
        if (!mainDescription) {
            mainDescription = "";
        }
        // Join the promotional text and app description, if both are present.
        let separator = "";
        if (promotionalText.length > 0 && mainDescription.length > 0) {
            separator = "\n\n";
        }
        const descriptionString = promotionalText + separator + mainDescription;
        if (descriptionString.length > 0) {
            const paragraph = new models.Paragraph(descriptionString);
            paragraph.isCollapsed = true;
            return paragraph;
        }
        return null;
    });
}
function linksFromData(objectGraph, data) {
    if (!data) {
        return null;
    }
    return validation.context("productDescriptionLinksFromData", () => {
        const links = [];
        const developerWebsiteUrl = contentAttributes.contentAttributeAsString(objectGraph, data, "websiteUrl");
        if (developerWebsiteUrl) {
            const action = new models.ExternalUrlAction(developerWebsiteUrl, false);
            const text = objectGraph.loc.string("PRODUCT_DEVELOPER_WEBSITE");
            links.push(new models.ProductPageLink(text, action, "safari" /* models.ProductPageLinkImageName.developer */));
        }
        const developerSupportUrl = contentAttributes.contentAttributeAsString(objectGraph, data, "supportURLForLanguage");
        if (developerSupportUrl) {
            const action = new models.ExternalUrlAction(developerSupportUrl, false);
            const text = objectGraph.loc.string("DEVELOPER_SUPPORT");
            links.push(new models.ProductPageLink(text, action, "questionmark.circle" /* models.ProductPageLinkImageName.support */));
        }
        return links;
    });
}
//# sourceMappingURL=description-shelf.js.map