import * as validation from "@jet/environment/json/validation";
import * as models from "../../../../api/models/index";
import * as serverData from "../../../../foundation/json-parsing/server-data";
import { shallowCopyOf } from "../../../../foundation/util/objects";
import * as productPageUtil from "../../product-page-util";
import * as categoryAnnotation from "./category-annotation";
import * as compatibilityAnnotation from "./compatibility-annotation";
import * as contentRatingAnnotation from "./content-rating-annotation";
import * as controllerAnnotation from "./controller-annotation";
import * as copyrightAnnotation from "./copyright-annotation";
import * as externalPurchasesAnnotation from "./external-purchases-annotation";
import * as externalBrowserAnnotation from "./external-browser-annotation";
import * as gameLicenseAnnotation from "./game-license-annotation";
import * as highMotionAnnotation from "./high-motion-annotation";
import * as languagesAnnotation from "./languages-annotation";
import * as locationAnnotation from "./location-annotation";
import * as productCapabilitiesAnnotation from "./product-capabilities-annotation";
import * as sellerAnnotation from "./seller-annotation";
import * as sizeAnnotation from "./size-annotation";
import * as spatialControllerAnnotation from "./spatial-controller-annotation";
import * as storefrontContentRatingAnnotation from "./storefront-content-rating-annotation";
import * as topInAppPurchasesAnnotation from "./top-in-app-purchases-annotation";
import * as versionAnnotation from "./version-annotation";
import * as contentAttributes from "../../../content/attributes";
import { isSome } from "@jet/environment";
import { externalPurchasesPlacementIsEnabled } from "../../../offers/external-purchases";
const standardList = [
    sellerAnnotation.createAnnotation,
    sizeAnnotation.createAnnotation,
    categoryAnnotation.createAnnotation,
    controllerAnnotation.createAnnotation,
    spatialControllerAnnotation.createAnnotation,
    compatibilityAnnotation.createAnnotation,
    externalBrowserAnnotation.createAnnotation,
    locationAnnotation.createAnnotation,
    languagesAnnotation.createAnnotation,
    highMotionAnnotation.createAnnotation,
    contentRatingAnnotation.createAnnotation,
    topInAppPurchasesAnnotation.createAnnotation,
    externalPurchasesAnnotation.createAnnotation,
    gameLicenseAnnotation.createAnnotation,
    copyrightAnnotation.createAnnotation,
];
const sellerPositionAdjustedList = [
    sizeAnnotation.createAnnotation,
    categoryAnnotation.createAnnotation,
    controllerAnnotation.createAnnotation,
    spatialControllerAnnotation.createAnnotation,
    compatibilityAnnotation.createAnnotation,
    externalBrowserAnnotation.createAnnotation,
    locationAnnotation.createAnnotation,
    languagesAnnotation.createAnnotation,
    highMotionAnnotation.createAnnotation,
    contentRatingAnnotation.createAnnotation,
    topInAppPurchasesAnnotation.createAnnotation,
    externalPurchasesAnnotation.createAnnotation,
    sellerAnnotation.createAnnotation,
    gameLicenseAnnotation.createAnnotation,
    copyrightAnnotation.createAnnotation,
];
const tvInformationList = [
    sellerAnnotation.createAnnotation,
    categoryAnnotation.createAnnotation,
    versionAnnotation.createAnnotation,
    sizeAnnotation.createAnnotation,
    gameLicenseAnnotation.createAnnotation,
    copyrightAnnotation.createAnnotation,
];
const tvSupportsList = [
    compatibilityAnnotation.createAnnotation,
    languagesAnnotation.createAnnotation,
    contentRatingAnnotation.createAnnotation,
    storefrontContentRatingAnnotation.createAnnotation,
];
const gamesList = [
    contentRatingAnnotation.createAnnotation,
    categoryAnnotation.createAnnotation,
    controllerAnnotation.createAnnotation,
    languagesAnnotation.createAnnotation,
    sizeAnnotation.createAnnotation,
];
function createAnnotations(objectGraph, annotationList, data, isFirstPartyHideableApp, isArcadeApp, suppressTitle, shelfMetrics, metadataHandler) {
    const annotations = [];
    for (const annotationFactory of annotationList) {
        const annotation = annotationFactory(objectGraph, data, isFirstPartyHideableApp, isArcadeApp, suppressTitle, shelfMetrics);
        if (isSome(annotation)) {
            // Don't overwrite existing items
            if (annotation.items_V3.length === 0) {
                annotation.items_V3 = translateItemsToV3(objectGraph, annotation);
            }
            annotation.expandAction = createExpandActionForAnnotation(objectGraph, annotation);
            annotations.push(annotation);
            if (metadataHandler) {
                metadataHandler(annotationFactory, annotation, annotations.length - 1);
            }
        }
    }
    return annotations;
}
export function translateItemsToV3(objectGraph, annotation) {
    const items_V3 = [];
    for (const item of annotation.items) {
        if (items_V3.length > 0) {
            items_V3.push({
                $kind: "spacer",
            });
        }
        if (isSome(item.headingArtworks)) {
            for (const artwork of item.headingArtworks) {
                items_V3.push({
                    $kind: "artwork",
                    artwork: artwork,
                });
            }
        }
        // Linkable Text section: covers heading, main text, and lists
        let markdownStyledText = "";
        if (isSome(item.heading)) {
            markdownStyledText += `**${item.heading}**`;
        }
        if (isSome(item.text)) {
            if (markdownStyledText.length > 0) {
                markdownStyledText += "\n";
            }
            markdownStyledText += item.text;
        }
        if (isSome(item.listText)) {
            if (markdownStyledText.length > 0) {
                markdownStyledText += "\n";
            }
            markdownStyledText += item.listText;
        }
        if (markdownStyledText.length > 0) {
            items_V3.push({
                $kind: "linkableText",
                linkableText: new models.LinkableText(new models.StyledText(markdownStyledText, "text/markdown")),
            });
        }
        if (isSome(item.textPairs)) {
            for (const textPair of item.textPairs) {
                items_V3.push({
                    $kind: "textPair",
                    leadingText: textPair[0],
                    trailingText: textPair[1],
                });
            }
        }
    }
    if (isSome(annotation.linkAction) && !objectGraph.client.isTV) {
        items_V3.push({
            $kind: "button",
            action: annotation.linkAction,
        });
    }
    return items_V3;
}
function standardInformationShelf(objectGraph, data, isFirstPartyHideableApp, shelfMetrics, isArcadeApp, metadataHandler) {
    return validation.context("standardInformationShelf", () => {
        let shelf;
        const list = shouldUseSellerPositionAdjustedList(objectGraph, data) ? sellerPositionAdjustedList : standardList;
        const informationItems = createAnnotations(objectGraph, list, data, isFirstPartyHideableApp, isArcadeApp, false, shelfMetrics, metadataHandler);
        if (informationItems.length > 0) {
            shelf = new models.Shelf("annotation");
            shelf.title = objectGraph.loc.string("ProductPage.Section.Information.Title");
            shelf.items = informationItems;
            if (!serverData.isNull(shelfMetrics)) {
                shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "information");
            }
        }
        return shelf;
    });
}
function gamesInformationShelf(objectGraph, data, isFirstPartyHideableApp, shelfMetrics, isArcadeApp, metadataHandler) {
    return validation.context("standardInformationShelf", () => {
        const informationItems = createAnnotations(objectGraph, gamesList, data, isFirstPartyHideableApp, isArcadeApp, true, shelfMetrics, metadataHandler);
        if (informationItems.length === 0) {
            return undefined;
        }
        const shelf = new models.Shelf("annotation");
        shelf.items = informationItems;
        return shelf;
    });
}
/** Checks if we should use the alternate position arrangement of seller info
 * - The Seller ICP Annotation feature is enabled
 * - The client platform is iOS
 * - The Internet Content Provider info is available
 * Or Seller info is available
 * @returns true if either of the checks above are true
 */
function shouldUseSellerPositionAdjustedList(objectGraph, data) {
    const icpInfo = contentAttributes.contentAttributeAsDictionary(objectGraph, data, "internetContentProviderInfo");
    const hasICPInfo = objectGraph.bag.enableSellerICPAnnotation && objectGraph.client.isiOS && isSome(icpInfo);
    const sellerInfo = contentAttributes.contentAttributeAsDictionary(objectGraph, data, "sellerInfo");
    const hasSellerInfo = objectGraph.bag.enableSellerInfo && isSome(sellerInfo);
    return hasICPInfo || hasSellerInfo;
}
function watchInformationShelf(objectGraph, data, isFirstPartyHideableApp, shelfMetrics) {
    return validation.context("standardInformationShelf", () => {
        let shelf;
        const list = shouldUseSellerPositionAdjustedList(objectGraph, data) ? sellerPositionAdjustedList : standardList;
        const informationItems = createAnnotations(objectGraph, list, data, isFirstPartyHideableApp, false, false, shelfMetrics, undefined);
        if (informationItems.length > 0) {
            shelf = new models.Shelf("annotation");
            shelf.items = informationItems;
            if (!serverData.isNull(shelfMetrics)) {
                shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "information");
            }
        }
        return shelf;
    });
}
function tvInformationShelf(objectGraph, data, isFirstPartyHideableApp, shelfMetrics, forceExpanded, isFreeProduct, metadataHandler) {
    return validation.context("tvInformationShelf", () => {
        let shelf;
        const informationItems = createAnnotations(objectGraph, tvInformationList, data, isFirstPartyHideableApp, false, false, shelfMetrics, metadataHandler);
        const supportsItems = productCapabilitiesAnnotation
            .createAnnotations(objectGraph, data, isFreeProduct)
            .concat(createAnnotations(objectGraph, tvSupportsList, data, isFirstPartyHideableApp, false, false, shelfMetrics, metadataHandler));
        const externalPurchasesEnabled = externalPurchasesPlacementIsEnabled(objectGraph, "product-page-info-section");
        const tvTopInAppPurchasesList = externalPurchasesEnabled
            ? [externalPurchasesAnnotation.createAnnotation]
            : [topInAppPurchasesAnnotation.createAnnotation];
        const topInAppPurchaseItems = createAnnotations(objectGraph, tvTopInAppPurchasesList, data, isFirstPartyHideableApp, false, true, shelfMetrics, metadataHandler);
        const annotationsGroups = [];
        if (informationItems.length > 0) {
            const annotationGroup = new models.AnnotationGroup(objectGraph.loc.string("ProductPage.Section.Information.Title"), informationItems, forceExpanded);
            annotationsGroups.push(annotationGroup);
        }
        if (supportsItems.length > 0) {
            const annotationGroup = new models.AnnotationGroup(objectGraph.loc.string("ProductPage.Section.Supports.Title"), supportsItems, forceExpanded);
            annotationsGroups.push(annotationGroup);
        }
        if (topInAppPurchaseItems.length > 0) {
            const inAppPurchasesTitle = externalPurchasesEnabled
                ? objectGraph.loc.string("ProductPage.Section.ExternalPurchases.Title")
                : objectGraph.loc.string("ProductPage.Section.TopInAppPurchases.Title");
            const annotationGroup = new models.AnnotationGroup(inAppPurchasesTitle, topInAppPurchaseItems, true);
            annotationsGroups.push(annotationGroup);
        }
        if (annotationsGroups.length > 0) {
            if (productPageUtil.isShelfBased(objectGraph)) {
                shelf = new models.Shelf("annotationGroup");
                shelf.items = annotationsGroups;
                shelf.background = {
                    type: "darkOverlay",
                };
                if (!serverData.isNull(shelfMetrics)) {
                    shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "information");
                }
            }
            else {
                shelf = new models.Shelf("informationContainer");
                shelf.items = [new models.InformationContainer(annotationsGroups)];
                if (!serverData.isNull(shelfMetrics)) {
                    shelfMetrics.addImpressionsToShelf(objectGraph, shelf, "information");
                }
            }
        }
        return shelf;
    });
}
/**
 * Creates the information shelf for the current client's deviceType.
 *
 * @param productData The product data
 * @param isFirstPartyHideableApp Indicates whether this app is first party and/or hideable
 * @param shelfMetrics The shelf metrics
 * @param metadataHandler The metadata handler used to associate an annotation with another element.
 * @param isArcadeApp Whether or not the product is an Arcade app.
 * @param isFreeProduct Whether the buy is for a free product.
 * @param forceExpanded Whether the items in the shelf should begin expanded
 */
export function create(objectGraph, productData, isFirstPartyHideableApp, shelfMetrics, metadataHandler, isArcadeApp, isFreeProduct, forceExpanded) {
    return validation.context("create", () => {
        if (preprocessor.GAMES_TARGET) {
            return gamesInformationShelf(objectGraph, productData, isFirstPartyHideableApp, shelfMetrics, isArcadeApp, metadataHandler);
        }
        let shelf;
        switch (objectGraph.client.deviceType) {
            case "watch":
                shelf = watchInformationShelf(objectGraph, productData, isFirstPartyHideableApp, shelfMetrics);
                break;
            case "tv":
                shelf = tvInformationShelf(objectGraph, productData, isFirstPartyHideableApp, shelfMetrics, forceExpanded, isFreeProduct, metadataHandler);
                break;
            default:
                shelf = standardInformationShelf(objectGraph, productData, isFirstPartyHideableApp, shelfMetrics, isArcadeApp, metadataHandler);
        }
        return shelf;
    });
}
/**
 * Creates a flow action for navigating to the expanded view of an annotation.
 *
 * @param objectGraph The current object graph
 * @param annotation The source annotation
 * @returns A flow action or null
 */
export function createExpandActionForAnnotation(objectGraph, annotation) {
    if (!objectGraph.client.isVision) {
        return undefined;
    }
    const flowAction = new models.FlowAction("annotationDetail");
    const detailAnnotation = shallowCopyOf(annotation);
    detailAnnotation.shouldAlwaysPresentExpanded = true;
    flowAction.pageData = detailAnnotation;
    return flowAction;
}
//# sourceMappingURL=annotations.js.map