import * as validation from "@jet/environment/json/validation";
import * as models from "../../api/models";
import * as serverData from "../../foundation/json-parsing/server-data";
import * as privacyHeaderShelf from "./privacy-header-shelf";
/**
 * Builder for the privacy footer shelf.
 */
export function create(objectGraph, data, pageInformation, locationTracker) {
    return validation.context("create", () => {
        if (serverData.isNullOrEmpty(data)) {
            return null;
        }
        if (objectGraph.client.isWatch) {
            return null;
        }
        const privacyTypes = privacyHeaderShelf.privacyTypesFromData(objectGraph, data, false, "detailPage", pageInformation, locationTracker);
        const privacyDataNotProvided = (privacyTypes.length === 1 && privacyTypes[0].identifier === "DATA_NOT_PROVIDED") ||
            privacyTypes.length === 0;
        if (privacyDataNotProvided && objectGraph.client.deviceType !== "tv") {
            return null;
        }
        const shelf = new models.Shelf("privacyFooter");
        const privacyFooter = privacyFooterFromData(objectGraph, data, pageInformation, locationTracker);
        shelf.items = [privacyFooter];
        return shelf;
    });
}
/**
 * Creates a privacy footer object.
 * @param data The data blob
 */
export function privacyFooterFromData(objectGraph, data, pageInformation, locationTracker) {
    return validation.context("privacyFooterFromData", () => {
        const bodyText = bodyTextFromData(objectGraph, data, pageInformation, locationTracker);
        const actions = actionsFromData(objectGraph, data, pageInformation, locationTracker);
        let privacyTypes = [];
        if (objectGraph.client.isTV) {
            privacyTypes = privacyHeaderShelf.privacyTypesFromData(objectGraph, data, false, "productPage", pageInformation, locationTracker);
        }
        return new models.PrivacyFooter(bodyText, actions, privacyTypes.length);
    });
}
/**
 * Creates the main body text for the footer.
 * @param data The data blob
 */
function bodyTextFromData(objectGraph, data, pageInformation, locationTracker) {
    let text;
    const learnMoreLinkText = objectGraph.loc.string("PRODUCT_PRIVACY_LEARN_MORE_LINK");
    const linkedSubstrings = {};
    const learnMoreAction = privacyHeaderShelf.createLearnMoreAction(objectGraph, pageInformation, locationTracker);
    if (objectGraph.client.isTV || serverData.isNull(learnMoreAction)) {
        text = objectGraph.loc.string("PRODUCT_PRIVACY_LEARN_MORE_NO_LINK");
    }
    else {
        text = objectGraph.loc.string("PRODUCT_PRIVACY_LEARN_MORE_TEMPLATE");
        text = text.replace("{learnMoreLink}", learnMoreLinkText);
        if (serverData.isNull(learnMoreAction)) {
            text = objectGraph.loc.string("PRODUCT_PRIVACY_LEARN_MORE_NO_LINK");
        }
        else {
            linkedSubstrings[learnMoreLinkText] = learnMoreAction;
        }
    }
    const textType = "text/plain";
    const styledText = new models.StyledText(text, textType);
    return new models.LinkableText(styledText, linkedSubstrings);
}
/**
 * Creates the actions for displaying in the privacy footer.
 */
function actionsFromData(objectGraph, data, pageInformation, locationTracker) {
    if (objectGraph.client.deviceType !== "tv") {
        return [];
    }
    const actions = [];
    const learnMoreAction = privacyHeaderShelf.createLearnMoreAction(objectGraph, pageInformation, locationTracker);
    if (serverData.isDefinedNonNull(learnMoreAction)) {
        actions.push(learnMoreAction);
    }
    const privacyDefinitionsAction = privacyHeaderShelf.createPrivacyDefinitionsAction(objectGraph, pageInformation, locationTracker);
    if (serverData.isDefinedNonNull(privacyDefinitionsAction)) {
        actions.push(privacyDefinitionsAction);
    }
    return actions;
}
//# sourceMappingURL=privacy-footer-shelf.js.map