import { unwrapOptional as unwrap } from "@jet/environment/types/optional";
import { attributeAsArrayOrEmpty } from "../foundation/media/attributes";
import { dataFromDataContainer } from "../foundation/media/data-structure";
import { deviceFamiliesFromData } from "./content/device-family";
/**
 * Retreive the current `PreviewPlatform`, in the format used to create a {@linkcode WithPlatform} `Intent`
 *
 * @example
 * makeIntentWithPlatform({
 *   ...getPlatform(objectGraph)
 *   ...otherProps
 * });
 */
export function getPlatform(objectGraph) {
    var _a;
    return {
        platform: (_a = objectGraph.activeIntent) === null || _a === void 0 ? void 0 : _a.previewPlatform,
    };
}
/**
 * Determine the most relevant {@linkcode PreviewPlatform} from a Media API response that
 * contains "device families", like an `app` or `app-bundle`
 */
export function inferPreviewPlatformFromDeviceFamilies(objectGraph, response) {
    const data = unwrap(dataFromDataContainer(objectGraph, response));
    const deviceFamilies = new Set(deviceFamiliesFromData(objectGraph, data, true));
    if (deviceFamilies.has("iphone")) {
        return "iphone";
    }
    if (deviceFamilies.has("ipad")) {
        return "ipad";
    }
    if (deviceFamilies.has("mac")) {
        return "mac";
    }
    if (deviceFamilies.has("realityDevice")) {
        return "vision";
    }
    if (deviceFamilies.has("tvos")) {
        return "tv";
    }
    if (deviceFamilies.has("watch")) {
        return "watch";
    }
    throw new Error("Could not infer platform from device families");
}
/**
 * Determine the most relevant {@linkcode PreviewPlatform} from a Media API response that
 * contains "editorial platforms"", like `editorial-items`
 */
export function inferPreviewPlatformFromEditorialPlatforms(objectGraph, container) {
    const data = unwrap(dataFromDataContainer(objectGraph, container));
    const editorialPlatforms = new Set(attributeAsArrayOrEmpty(data, "editorialPlatforms"));
    if (editorialPlatforms.has("iphone")) {
        return "iphone";
    }
    if (editorialPlatforms.has("ipad")) {
        return "ipad";
    }
    if (editorialPlatforms.has("desktop")) {
        return "mac";
    }
    if (editorialPlatforms.has("realitydevice")) {
        return "vision";
    }
    if (editorialPlatforms.has("watch")) {
        return "watch";
    }
    if (editorialPlatforms.has("appletv")) {
        return "tv";
    }
    throw new Error("Could not infer preview platform from editorial platforms");
}
/**
 * Set the `previewPlatform` query param on the {@linkcode request}, if necessary
 *
 * This should only be used with requests being made for the following resources:
 * - "Editorial" (Grouping Page, Editorial Page, Today, etc)
 * - "Search"
 */
export function setPreviewPlatform(objectGraph, request) {
    var _a;
    return ((_a = objectGraph.activeIntent) === null || _a === void 0 ? void 0 : _a.previewPlatform)
        ? request.addingQuery("previewPlatform", objectGraph.activeIntent.platform).addingQuery("platform", "web")
        : request;
}
//# sourceMappingURL=preview-platform.js.map