import * as serverData from "../../foundation/json-parsing/server-data";
import { demoteByEngagements } from "@amp/amd-apps";
// The AMD key used to retrieve the engagement events from the app store.
export const AMSEngagementAppStoreEventKey = "appStore.getEngagementEvents";
/**
 * personalize the dataItems based on the impression data for that shelf.
 *
 * @param objectGraph The object graph.
 * @param dataItems The data items to personalize.
 * @param impressionData The impression data to use for personalization.
 * @returns The personalized data items.
 */
export function personalizeDataItems(dataItems, impressionData, shelfRecoMetricsData) {
    const shelfAlgoId = shelfRecoMetricsData["reco_algoId"];
    if (serverData.isNullOrEmpty(shelfAlgoId) ||
        serverData.isNullOrEmpty(dataItems) ||
        serverData.isNullOrEmpty(impressionData[shelfAlgoId])) {
        return dataItems;
    }
    // First create Candidate objects from the data items.
    const candidates = dataItems.map((dataItem) => {
        var _a;
        const adamId = serverData.asNumber(dataItem.id);
        const score = (_a = serverData.asNumber(dataItem, "meta.personalizationData.score")) !== null && _a !== void 0 ? _a : 0;
        const candidate = { identifier: adamId, score: score };
        return candidate;
    });
    // Demote the candidates based on the impression data.
    const shelfRecoData = impressionData[shelfAlgoId];
    const rerankedCandidates = demoteByEngagements(candidates, shelfRecoData);
    // Create a lookup map from dataItems to allow faster rearranging
    const dictionary = dataItems.reduce((acc, item) => ({ ...acc, [item.id]: item }), {});
    // create a new array based on the rearranged Candidate array
    const rearranged = rerankedCandidates.map((candidate) => dictionary[candidate.identifier.toString()]);
    return rearranged;
}
/**
 * Takes the AMD response and creates a map of EngagementEvent per shelf.
 *
 * @param objectGraph The object graph.
 * @param data The data to convert.
 * @returns The map of EngagementEvent per shelf.
 */
export function impressionEventsFromData(objectGraph, data) {
    // go through all the impressionData and return the EgagementData
    if (!serverData.isDefinedNonNullNonEmpty(data)) {
        return {};
    }
    // Impression data is a map of shelf ids to impression arrays.
    // We want to keep this relationship so only impressions for a given shelf are returned.
    const impressionData = serverData.asDictionary(data, "data.engagementEvents.impression");
    const convertedMap = {};
    // Iterate over each key in the original map and convert the data into EngagementEvents which can be passed to the demoteByEngagements function.
    for (const key in impressionData) {
        if (serverData.isDefinedNonNullNonEmpty(key)) {
            const impressions = serverData.asArrayOrEmpty(impressionData, key);
            convertedMap[key] = impressions.map((impression) => {
                const adamId = serverData.asNumber(impression["adamId"]);
                const timestamp = serverData.asNumber(impression["eventTimeMillis"]);
                const event = { adamIdentifier: adamId, timestamp: timestamp };
                return event;
            });
        }
    }
    return convertedMap;
}
/**
 * Convenience function for determining if data personalization is available.
 */
export function isImpressionDemotionAvailable(objectGraph) {
    return objectGraph.client.isiOS && objectGraph.bag.enableRecoOnDeviceReordering;
}
//# sourceMappingURL=on-device-impression-demotion.js.map