import * as mediaDataStructure from "../../../../foundation/media/data-structure";
import * as models from "../../../../api/models";
import * as searchShelves from "../../../search/content/search-shelves";
import { impressionOptions } from "../impressions";
import { asString } from "../../../../foundation/json-parsing/server-data";
import { relationship } from "../../../../foundation/media/relationships";
import { isNothing } from "@jet/environment/types/optional";
/**
 * Generates the metrics impressions options for the search shelf
 * @param objectGraph The App Store Object Graph
 * @param data The shelf data object
 * @param shelfAttributes The shelf's attributes
 * @param searchPageContext The context for the page containing the shelf
 * @returns The metrics options for the shelf
 */
export function createMetricsOptionsForGenericSearchPageShelf(objectGraph, data, shelfAttributes, searchPageContext) {
    var _a, _b, _c;
    /// On shelves, the actual reco metrics are on the content and not the data blob itself
    const recoMetricsDataContainer = relationship(data, "contents");
    const recoMetricsData = recoMetricsDataContainer === null
        ? undefined
        : (_a = mediaDataStructure.metricsFromMediaApiObject(recoMetricsDataContainer)) !== null && _a !== void 0 ? _a : undefined;
    let impressionsIdType = "its_contentId";
    if (searchPageContext.pageType === searchShelves.SearchPageType.ChartsAndCategories) {
        impressionsIdType = "static";
    }
    const shelfMetricsOptions = {
        id: shelfAttributes.id,
        kind: null,
        softwareType: null,
        targetType: "swoosh",
        title: (_b = shelfAttributes.title) !== null && _b !== void 0 ? _b : "",
        pageInformation: searchPageContext.metricsPageInformation,
        locationTracker: searchPageContext.metricsLocationTracker,
        idType: impressionsIdType,
        fcKind: undefined,
        canonicalId: (_c = asString(data.meta, "canonicalId")) !== null && _c !== void 0 ? _c : undefined,
        recoMetricsData: recoMetricsData,
    };
    return shelfMetricsOptions;
}
/**
 * Generates the impressions metrics options for the search chart or category
 * @param objectGraph The App Store Object Graph
 * @param model The chart or category model
 * @param modelData The chart or category model data object
 * @param searchShelfContext The context for the shelf containing the chart or category
 * @returns The metrics options for the chart or category model
 */
export function createMetricsOptionsForChartOrCategory(objectGraph, model, modelData, searchShelfContext) {
    var _a;
    const chartModelMetricsOptions = {
        ...searchShelfContext.metricsOptions,
        ...impressionOptions(objectGraph, modelData, model.title, searchShelfContext.metricsOptions),
        recoMetricsData: (_a = mediaDataStructure.metricsFromMediaApiObject(modelData)) !== null && _a !== void 0 ? _a : undefined,
        targetType: metricsTargetTypeForChartOrCategory(model.density),
        idType: "its_id",
    };
    return chartModelMetricsOptions;
}
/**
 * Generates the click metrics options for the search chart or category
 * @param objectGraph The App Store Object Graph
 * @param model The chart or category model
 * @param modelData The chart or category model data object
 * @param searchShelfContext The context for the shelf containing the chart or category
 * @returns The metrics options for the chart or category model
 */
export function createClickMetricsOptionsForChartOrCategory(objectGraph, modelDensity, modelData, searchShelfContext) {
    var _a;
    const chartModelMetricsOptions = {
        pageInformation: searchShelfContext.metricsOptions.pageInformation,
        locationTracker: searchShelfContext.metricsOptions.locationTracker,
        recoMetricsData: (_a = mediaDataStructure.metricsFromMediaApiObject(modelData)) !== null && _a !== void 0 ? _a : undefined,
        targetType: metricsTargetTypeForChartOrCategory(modelDensity),
        id: modelData.id,
    };
    return chartModelMetricsOptions;
}
/**
 * Gets the matching metrics target type for the chart or category model
 * @param model The model we want the target type for
 * @returns The target type for the model
 */
function metricsTargetTypeForChartOrCategory(modelDensity) {
    if (isNothing(modelDensity)) {
        return "tile";
    }
    switch (modelDensity) {
        case models.GenericSearchPageShelfDisplayStyleDensity.Density1:
            return "tile";
        case models.GenericSearchPageShelfDisplayStyleDensity.Density2:
            return "pill";
        default:
            return "tile";
    }
}
//# sourceMappingURL=search-shelves.js.map