import * as models from "../../api/models";
import * as serverData from "../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../foundation/media/attributes";
import * as objects from "../../foundation/util/objects";
import * as metricsHelpersClicks from "../metrics/helpers/clicks";
/**
 *                                      "External Deep Link" feature description
 *
 *  This feature is one where a particular offer will behave like a normal offer when the app is not openable, but in
 *  which the openable state for the offer actually deep links to the third party app in question. This deep link uses
 *  a universal link that is plumbed through by Media API; the deep link is submitted and approved through
 *  App Store Connect and will be interpreted by the third party app in accordance with the external action.
 *
 *  This feature's name is prefixed by the term "external" to differentiate from deep linking into the App Store app
 *  itself (such as deep linking into the search tab via Siri or the user's purchases through a finance page).
 */
/// The query parameter name to use when attaching the external deep link to a product URL.
export const externalDeepLinkQueryParameter = "externalDeepLinkUrl";
/// Query parameter name to use when attaching eligibility information for cpp deep links.
export const cppDeepLinkDisabledQueryParameter = "isCppDeepLinkDisabled";
/// Query parameter name to use when attaching the aligned region deep link to a product URL.
export const alignedRegionDeepLinkQueryParameter = "alignedRegionDeepLinkUrl";
/**
 * Pulls out the external deep link url from the given data.
 * @param {Data} data The data for the app that has the deep link.
 * @return {string | null} The external deep link url, if it exists.
 */
export function deepLinkUrlFromData(objectGraph, data) {
    if (!serverData.isDefinedNonNull(data)) {
        return null;
    }
    return mediaAttributes.attributeAsString(data, "customUrl");
}
/**
 * Wraps an action in a state action that executes an external deep link when the app is openable.
 * @param {Action} action The action for the product that is being deep linked into.
 * @param {string | null} adamId The adam ID of the app.
 * @param {string | null} bundleId The bundle ID of the app.
 * @param {string | null} deepLinkUrl The url for the deep link action.
 * @param {boolean} includeBetaApps Whether to include beta apps in the resulting action.
 * @param metricsClickOptions The metrics click options for the action.
 * @return {Action} The action to use for the app that is deep linked.
 */
export function deepLinkActionWrappingAction(objectGraph, action, adamId, bundleId, deepLinkUrl, includeBetaApps, metricsClickOptions) {
    // Only wrap in OfferStateAction on devices that are supported when there's a deep link present.
    if (!serverData.isDefinedNonNullNonEmpty(deepLinkUrl) && objectGraph.client.deviceType !== "mac") {
        return action;
    }
    let openAction;
    if (objectGraph.client.isiOS) {
        const openAppAction = new models.OpenAppAction(adamId, "app");
        openAction = new models.AppLaunchTrampolineAction(bundleId, deepLinkUrl, openAppAction);
    }
    else {
        openAction = new models.ExternalUrlAction(deepLinkUrl);
    }
    const clickOptions = objects.shallowCopyOf(metricsClickOptions);
    clickOptions.actionType = "open";
    clickOptions.actionDetails = { actionUrl: deepLinkUrl };
    metricsHelpersClicks.addClickEventToAction(objectGraph, openAction, clickOptions, true, "button");
    if (action instanceof models.OfferStateAction) {
        action.openAction = openAction;
        action.includeBetaApps = includeBetaApps;
        return action;
    }
    else {
        const stateAction = new models.OfferStateAction(adamId, action);
        stateAction.openAction = openAction;
        stateAction.includeBetaApps = includeBetaApps;
        return stateAction;
    }
}
//# sourceMappingURL=external-deep-link.js.map