import * as models from "../../../api/models";
import * as serverData from "../../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../../foundation/media/attributes";
import * as mediaDataStructure from "../../../foundation/media/data-structure";
import * as mediaPlatformAttributes from "../../../foundation/media/platform-attributes";
import * as mediaRelationship from "../../../foundation/media/relationships";
import * as color from "../../../foundation/util/color-util";
import * as breakoutsCommon from "../../arcade/breakouts-common";
import * as contentAttributes from "../../content/attributes";
import * as content from "../../content/content";
import * as flowPreview from "../../content/flow-preview";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as metricsHelpersImpressions from "../../metrics/helpers/impressions";
import * as metricsHelpersLocation from "../../metrics/helpers/location";
import * as article from "../../today/article";
import { GroupingShelfController } from "./grouping-shelf-controller";
import * as groupingShelfControllerCommon from "./grouping-shelf-controller-common";
export class GroupingSmallBreakoutShelfController extends GroupingShelfController {
    // region Constructors
    constructor() {
        super("GroupingSmallBreakoutShelfController");
        this.supportedFeaturedContentIds = new Set([480 /* groupingTypes.FeaturedContentID.AppStore_Breakout */]);
    }
    // endregion
    // region GroupingShelfController
    _supports(objectGraph, mediaApiData, featuredContentId, nativeGroupingShelfId) {
        if (!super._supports(objectGraph, mediaApiData, featuredContentId, nativeGroupingShelfId)) {
            return false;
        }
        const breakoutStyle = mediaAttributes.attributeAsString(mediaApiData, "displayStyle");
        return breakoutStyle === "small";
    }
    // endregion
    // region Shelf Creation Prerequisites
    /**
     * For a given mediaApiData extract the actual shelfContents array needed to render this shelf
     *
     * @param mediaApiData The outer shelfContents object containing the shelf contents
     */
    initialShelfDataFromGroupingMediaApiData(objectGraph, mediaApiData) {
        return { shelfContents: mediaRelationship.relationshipCollection(mediaApiData, "contents") };
    }
    /**
     * For a given url that this controller handles, we should return a promise that will result in the `ShelfData`
     * needed to render this shelf
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfUrl The url that this controller handled on a secondary fetch
     * @param parameters The extracted parameters from the shelf url
     */
    async secondaryShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters) {
        return await GroupingShelfController.secondaryGroupingShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters);
    }
    /**
     * For a given mediaApiData create an updated shelf token that contains all the additional data for this specific shelf type
     *
     * @param objectGraph The App Store dependency graph
     * @param baseShelfToken The base grouping shelf token created by the grouping-controller
     * @param mediaApiData The outer data object containing the FC properties and data
     * @param groupingParseContext The parse context for the grouping page so far
     */
    shelfTokenFromBaseTokenAndMediaApiData(objectGraph, mediaApiData, baseShelfToken, groupingParseContext) {
        return baseShelfToken;
    }
    // endregion
    // region Metrics
    shouldImpressShelf() {
        return false;
    }
    // endregion
    // region Shelf Creation
    /**
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfToken The shelf shelfToken for this current shelf creation request
     * @param shelfData The media api shelfContents array for this shelf
     * @param groupingParseContext The parse context used to generate the grouping page on the initial page load,
     * this will be missing when this controller renders a secondary or incomplete shelf fetch.
     */
    _createShelf(objectGraph, shelfToken, shelfData, groupingParseContext) {
        const items = [];
        for (const data of shelfData.shelfContents) {
            if (serverData.isNull(data.attributes) || groupingShelfControllerCommon.shouldDefer(shelfToken)) {
                shelfToken.isDeferring = true;
                shelfToken.remainingItems.push(data);
                continue;
            }
            const primaryContent = content.primaryContentForData(objectGraph, data);
            if (breakoutsCommon.requiresPrimaryContent(objectGraph, data) &&
                !mediaAttributes.hasAttributes(primaryContent)) {
                shelfToken.isDeferring = true;
                shelfToken.remainingItems.push(data);
                shelfToken.relationshipToFetch = "primary-content";
                continue;
            }
            const metricsOptions = {
                targetType: "smallBreakout",
                pageInformation: shelfToken.metricsPageInformation,
                locationTracker: shelfToken.metricsLocationTracker,
                recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(data),
            };
            // If this EI can and should show the expected release date of an arcade app, override the badge.
            let badgeTitle;
            const fallbackLabel = mediaAttributes.attributeAsString(data, "label");
            if (contentAttributes.contentAttributeAsBooleanOrFalse(objectGraph, data, "showExpectedReleaseDate")) {
                badgeTitle = objectGraph.loc.uppercased(content.dynamicPreorderDateFromData(objectGraph, primaryContent, fallbackLabel));
            }
            else {
                badgeTitle = fallbackLabel;
            }
            let badge = { type: "none" };
            if ((badgeTitle === null || badgeTitle === void 0 ? void 0 : badgeTitle.length) > 0) {
                badge = {
                    type: "text",
                    title: badgeTitle,
                };
            }
            const title = content.editorialNotesFromData(objectGraph, data, "name") ||
                contentAttributes.contentAttributeAsString(objectGraph, primaryContent, "name");
            const description = content.editorialNotesFromData(objectGraph, data, "short") ||
                contentAttributes.contentAttributeAsString(objectGraph, primaryContent, "tagline");
            const artwork = content.iconFromData(objectGraph, primaryContent, {
                useCase: 5 /* content.ArtworkUseCase.ArcadeSmallBreakout */,
                withJoeColorPlaceholder: true,
            });
            const attributePlatform = contentAttributes.bestAttributePlatformFromData(objectGraph, data);
            const artworkData = mediaPlatformAttributes.platformAttributeAsDictionary(primaryContent, attributePlatform, "artwork");
            const backgroundColor = color.fromHex(serverData.asString(artworkData, "textColor4")) || (artwork === null || artwork === void 0 ? void 0 : artwork.backgroundColor);
            const details = new models.BreakoutDetails(title, description, badge, null, breakoutsCommon.detailBackgroundStyleFromColor(objectGraph, backgroundColor), null);
            const smallBreakout = new models.SmallBreakout(details, artwork, backgroundColor);
            const impressionOptions = metricsHelpersImpressions.impressionOptions(objectGraph, data, smallBreakout.details.title, metricsOptions);
            const productData = article.productDataFromArticle(objectGraph, data);
            const isPreorder = contentAttributes.contentAttributeAsBooleanOrFalse(objectGraph, productData, "isPreorder");
            impressionOptions.isPreorder = isPreorder;
            metricsHelpersImpressions.addImpressionFields(objectGraph, smallBreakout, impressionOptions);
            // Push the breakout here so that the click action has the breakout in its location
            // but we do not want to add it to the overall location tracker, so pop it right after adding it to the button
            // action
            // <rdar://problem/60883269> Metrics: Arcade: Container values requested in Location field
            metricsHelpersLocation.pushContentLocation(objectGraph, impressionOptions, smallBreakout.details.title);
            const breakoutAction = breakoutsCommon.actionFromData(objectGraph, data);
            const breakoutClickOptions = {
                pageInformation: shelfToken.metricsPageInformation,
                locationTracker: shelfToken.metricsLocationTracker,
                recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(data),
                targetType: "button",
                id: data.id,
            };
            metricsHelpersClicks.addClickEventToAction(objectGraph, breakoutAction, breakoutClickOptions);
            smallBreakout.details.callToActionButtonAction = breakoutAction;
            smallBreakout.clickAction = breakoutAction;
            metricsHelpersLocation.popLocation(metricsOptions.locationTracker);
            // Set flow preview actions
            smallBreakout.flowPreviewActionsConfiguration =
                flowPreview.flowPreviewActionsConfigurationForProductFromData(objectGraph, data, true, shelfToken === null || shelfToken === void 0 ? void 0 : shelfToken.clientIdentifierOverride, breakoutAction, metricsOptions, breakoutClickOptions);
            items.push(smallBreakout);
        }
        const shelf = new models.Shelf("smallBreakout");
        shelf.isHorizontal = false;
        shelf.items = items;
        shelf.url = groupingShelfControllerCommon.createShelfTokenUrlIfNecessaryForShelf(objectGraph, shelf, shelfToken);
        if (groupingParseContext.shelves.length === 0) {
            shelf.presentationHints = { isFirstShelf: true };
        }
        return shelf;
    }
}
//# sourceMappingURL=grouping-small-breakout-shelf-controller.js.map