import * as models from "../../../api/models";
import { Parameters, Path, Protocol } from "../../../foundation/network/url-constants";
import * as urls from "../../../foundation/network/urls";
import * as metricsHelpersClicks from "../../metrics/helpers/clicks";
import * as metricsHelpersImpressions from "../../metrics/helpers/impressions";
import * as metricsHelpersLocation from "../../metrics/helpers/location";
import { GroupingShelfController } from "./grouping-shelf-controller";
import * as groupingShelfControllerCommon from "./grouping-shelf-controller-common";
import { ActionMetrics } from "../../../api/models";
import { makeGameCenterHeader } from "../../arcade/arcade-common";
export class GroupingGameCenterActivityFeedController extends GroupingShelfController {
    // region Constructors
    constructor() {
        super("GroupingGameCenterActivityFeedController");
        this.batchGroupKey = "gameCenter";
        this.supportedFeaturedContentIds = new Set([
            548 /* groupingTypes.FeaturedContentID.AppStore_GameCenterActivityFeedMarker */,
        ]);
    }
    // endregion
    // region Shelf Builder
    shelfRoute(objectGraph) {
        return [
            ...super.shelfRoute(objectGraph),
            {
                protocol: Protocol.internal,
                path: `/${Path.grouping}/${Path.shelf}/{token}`,
                query: [Parameters.isGameCenterActivityFeedShelf],
            },
        ];
    }
    // endregion
    // region Shelf Creation Prerequisites
    /**
     * For a given mediaApiData extract the actual shelfContents array needed to render this shelf
     *
     * @param mediaApiData The outer shelfContents object containing the shelf contents
     */
    initialShelfDataFromGroupingMediaApiData(objectGraph, mediaApiData) {
        return {
            shelfContents: [],
            activities: [],
        };
    }
    /**
     * For a given url that this controller handles, we should return a promise that will result in the `ShelfData`
     * needed to render this shelf
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfUrl The url that this controller handled on a secondary fetch
     * @param parameters The extracted parameters from the shelf url
     */
    async secondaryShelfDataForShelfUrl(objectGraph, shelfUrl, shelfToken, parameters) {
        const filter = this.gameCategoryFilter(shelfToken.gamesFilter);
        const activityLimit = 20;
        return await objectGraph.gameCenter.fetchActivityFeedCards(filter, activityLimit).then((activities) => {
            return {
                shelfContents: [],
                activities: activities,
            };
        });
    }
    /**
     * For a given mediaApiData create an updated shelf token that contains all the additional data for this specific shelf type
     *
     * @param objectGraph The App Store dependency graph
     * @param baseShelfToken The base grouping shelf token created by the grouping-controller
     * @param mediaApiData The outer data object containing the FC properties and data
     * @param groupingParseContext The parse context for the grouping page so far
     */
    shelfTokenFromBaseTokenAndMediaApiData(objectGraph, mediaApiData, baseShelfToken, groupingParseContext) {
        return baseShelfToken;
    }
    incompleteShelfFetchStrategy(objectGraph) {
        return models.IncompleteShelfFetchStrategy.OnPageLoad;
    }
    // endregion
    // region Shelf Creation
    /**
     *
     * @param objectGraph The App Store dependency graph
     * @param shelfToken The shelf shelfToken for this current shelf creation request
     * @param shelfData The media api shelfContents array for this shelf
     * @param groupingParseContext The parse context used to generate the grouping page on the initial page load,
     * this will be missing when this controller renders a secondary or incomplete shelf fetch.
     */
    _createShelf(objectGraph, shelfToken, shelfData, groupingParseContext) {
        if (shelfToken.isFirstRender) {
            return this.pendingActivityFeedShelfForGrouping(objectGraph, shelfToken, groupingParseContext === null || groupingParseContext === void 0 ? void 0 : groupingParseContext.isArcadePage);
        }
        else {
            return this.activityFeedShelfForGrouping(objectGraph, shelfData, shelfToken, groupingParseContext === null || groupingParseContext === void 0 ? void 0 : groupingParseContext.isArcadePage);
        }
    }
    pendingActivityFeedShelfForGrouping(objectGraph, shelfToken, isArcadePage) {
        if (objectGraph.client.deviceType !== "phone" && objectGraph.client.deviceType !== "pad") {
            return null;
        }
        const shelf = this.activityFeedShelfForGrouping(objectGraph, {
            shelfContents: [],
            activities: [],
        }, shelfToken, isArcadePage);
        const groupingShelfUrl = urls.URL.from(groupingShelfControllerCommon.groupingShelfUrl(shelfToken));
        shelf.url = groupingShelfUrl.param(Parameters.isGameCenterActivityFeedShelf, "true").build();
        shelf.isHidden = shelf.items.length === 0;
        shelf.batchGroup = this.batchGroupKey;
        return shelf;
    }
    activityFeedShelfForGrouping(objectGraph, shelfData, shelfToken, isArcadePage) {
        if (objectGraph.client.deviceType !== "phone" && objectGraph.client.deviceType !== "pad") {
            return null;
        }
        const shelf = this.activityFeedShelf(objectGraph, shelfData.activities, shelfToken, isArcadePage);
        const title = objectGraph.loc.string("Arcade.ActivityFeed.RecentActivity");
        shelf.header = makeGameCenterHeader(objectGraph, title);
        // Connect the shelf's seeAllAction
        groupingShelfControllerCommon.replaceShelfSeeAllAction(objectGraph, shelf, shelf.seeAllAction);
        shelf.batchGroup = this.batchGroupKey;
        // Hide when empty.
        shelf.isHidden = shelf.items.length === 0;
        return shelf;
    }
    // region Helpers
    activityFeedShelf(objectGraph, activities, token, isArcadePage = false) {
        const shelf = new models.Shelf("gameCenterActivityFeedCard");
        shelf.isHorizontal = true;
        shelf.mergeWhenFetched = true;
        shelf.batchGroup = this.batchGroupKey;
        shelf.items = activities;
        shelf.isHidden = shelf.items.length === 0;
        activities.forEach((item, index) => {
            const metricsImpressionOptions = {
                id: "friendActivity",
                idType: "static",
                targetType: "chiclet",
                kind: null,
                softwareType: isArcadePage ? "Arcade" : null,
                title: "",
                pageInformation: token.metricsPageInformation,
                locationTracker: token.metricsLocationTracker,
            };
            metricsHelpersImpressions.addImpressionFields(objectGraph, item, metricsImpressionOptions);
            // Create action metrics for click events.
            const profileActionMetrics = new ActionMetrics();
            const profileAvatarActionMetrics = new ActionMetrics();
            const leaderboardActionMetrics = new ActionMetrics();
            const achievementActionMetrics = new ActionMetrics();
            const appActionMetrics = new ActionMetrics();
            // Create targetId and metrics array.
            const targetIdAndMetricsArray = [
                {
                    targetId: "playerName",
                    metrics: profileActionMetrics,
                },
                {
                    targetId: "profileImage",
                    metrics: profileAvatarActionMetrics,
                },
                {
                    targetId: "leaderboardAchievement",
                    metrics: leaderboardActionMetrics,
                },
                {
                    targetId: "achievement",
                    metrics: achievementActionMetrics,
                },
                {
                    targetId: item.adamID || "gameIcon",
                    metrics: appActionMetrics,
                },
            ];
            // Loop through `targetIdAndMetricsArray` and call `addClickEventToActivityFeedMetrics` for each metrics.
            targetIdAndMetricsArray.forEach((targetIdAndMetricsDictionary) => metricsHelpersClicks.addClickEventToActivityFeedMetrics(objectGraph, targetIdAndMetricsDictionary.metrics, token.title, targetIdAndMetricsDictionary.targetId, {
                pageInformation: token.metricsPageInformation,
                locationTracker: token.metricsLocationTracker,
            }));
            // Assign action metrics to `item`.
            item.profileActionMetrics = profileActionMetrics;
            item.profileAvatarActionMetrics = profileAvatarActionMetrics;
            item.leaderboardActionMetrics = leaderboardActionMetrics;
            item.achievementActionMetrics = achievementActionMetrics;
            item.appActionMetrics = appActionMetrics;
            // Proceed to next position.
            metricsHelpersLocation.nextPosition(token.metricsLocationTracker);
        });
        // Setup see all action
        let seeAllAction;
        if (!objectGraph.featureFlags.isGSEUIEnabled("de7bbd8e")) {
            seeAllAction = new models.GameCenterDashboardAction();
            seeAllAction.title = objectGraph.loc.string("Arcade.ActivityFeed.AllActivity", objectGraph.loc.string("ACTION_SEE_ALL"));
            metricsHelpersClicks.addClickEventToSeeAllAction(objectGraph, seeAllAction, null, {
                pageInformation: token.metricsPageInformation,
                locationTracker: token.metricsLocationTracker,
            });
        }
        shelf.seeAllAction = seeAllAction;
        return shelf;
    }
    // endregion
    /**
     * Convert GameCategoryFilter to GamesFilter. Ideally these would be the same, but seed 1 has already left the station.
     * @param gamesFilter
     */
    gameCategoryFilter(gamesFilter) {
        if (gamesFilter === "nonArcade") {
            return "nonarcade";
        }
        return gamesFilter;
    }
    // endregion
    // region Metrics
    shelfMetricsOptionsFromBaseMetricsOptions(objectGraph, shelfToken, baseMetricsOptions) {
        return {
            ...baseMetricsOptions,
            title: "Friend Activity",
            badges: {
                gameCenter: true,
            },
            idType: "shelf_id",
        };
    }
}
//# sourceMappingURL=grouping-game-center-activity-feed-shelf-controller.js.map