import { Request, defaultAdditionalPlatformsForClient, defaultSparseLimitForClient, } from "../../foundation/media/data-fetching";
import { configureContingentItemsForGroupingRequest, configureOfferItemsForMediaRequest, configureTagsForMediaRequest, } from "../builders/url-mapping-utils";
import { shouldFetchCustomAttributes } from "../product-page/product-page-variants";
import { appContingentItemsAreEnabled, appEventsAreEnabled, appOfferItemsAreEnabled, } from "../app-promotions/app-promotions-common";
import { areAppTagsEnabled } from "../util/app-tags-util";
import * as mediaRequestUtils from "../../common/builders/url-mapping-utils";
import { shouldUsePrerenderedIconArtwork } from "../content/content";
import { AppEventsAttributes } from "../../gameservicesui/src/foundation/media-api/requests/recommendation-request-types";
/**
 * Create a "base" Media API {@linkcode Request} object for a "Grouping" page
 *
 * This is used to provision an initial {@linkcode Request} object that will be further
 * modified later to include the specific page that should be fetched
 *
 * @see {@linkcode prepareGroupingPageRequest} to complete configuration of the `Request` object
 */
export function makeBaseGroupingPageRequest(objectGraph) {
    // TODO: Update this code to work with genreId when this radar is complete
    // <rdar://problem/37676122> Media Api: Need ability to lookup grouping resource by genreId
    // TODO: Remove isAppleWatchSupported, once the following radar is resolved
    // <rdar://problem/38923866> Media API: Include isAppleWatchSupported in secondary lookups
    // Add this back in 18E when grouping supports additional platforms
    // fixed with <rdar://problem/38899888> App Store Personalization: grouping: remove tabs query param for Apps realm
    // .includingAdditionalPlatforms(mediaDataFetching.defaultAdditionalPlatformsForClient())
    const attributes = [
        "editorialArtwork",
        "editorialVideo",
        "isAppleWatchSupported",
        "requiredCapabilities",
        "expectedReleaseDateDisplayFormat",
        "showExpectedReleaseDate",
        "badge-content",
    ];
    if (objectGraph.appleSilicon.isSupportEnabled) {
        attributes.push("macRequiredCapabilities");
    }
    if (objectGraph.client.isMac) {
        attributes.push("hasMacIPAPackage");
    }
    if (objectGraph.bag.enableUpdatedAgeRatings) {
        attributes.push("ageRating");
    }
    if (shouldUsePrerenderedIconArtwork(objectGraph)) {
        attributes.push("iconArtwork");
    }
    const mediaApiRequest = new Request(objectGraph)
        .forType("groupings")
        .includingAgeRestrictions()
        .includingAttributes(attributes)
        .includingRelationshipsForUpsell(true);
    if (areAppTagsEnabled(objectGraph, "grouping")) {
        mediaRequestUtils.configureTagsForMediaRequest(mediaApiRequest);
    }
    return mediaApiRequest;
}
/**
 * Complete the configuration of a "Grouping Page" {@linkcode Request}
 */
export function prepareGroupingPageRequest(objectGraph, request) {
    request.includingAdditionalPlatforms(defaultAdditionalPlatformsForClient(objectGraph));
    request.includingAgeRestrictions();
    request.usingCustomAttributes(shouldFetchCustomAttributes(objectGraph));
    // <rdar://53420717> For performance reasons, we limit shelves to 3 items on watchOS
    if (objectGraph.client.isWatch) {
        request.addingRelationshipLimit("contents", 3);
    }
    if (objectGraph.client.isWeb) {
        // The "web" client needs to load *all* of the data for SEO purposes
        // Without this, shelves will render with some missing items
        request.withSparseCount(40);
        const sparseLimit = defaultSparseLimitForClient(objectGraph);
        if (sparseLimit) {
            request.includingScopedSparseLimit("editorial-elements:contents", sparseLimit);
        }
    }
    request.includingMacOSCompatibleIOSAppsWhenSupported();
    if (appEventsAreEnabled(objectGraph)) {
        request.enablingFeature("appEvents");
        request.includingMetaKeys("editorial-elements:contents", ["personalizationData", "cppData"]);
        request.includingScopedAttributes("app-events", AppEventsAttributes);
        request.includingScopedRelationships("app-events", ["app"]);
    }
    if (areAppTagsEnabled(objectGraph, "grouping")) {
        configureTagsForMediaRequest(request);
    }
    if (appContingentItemsAreEnabled(objectGraph)) {
        request.enablingFeature("contingentItems");
        configureContingentItemsForGroupingRequest(request);
    }
    if (appOfferItemsAreEnabled(objectGraph)) {
        request.enablingFeature("offerItems");
        configureOfferItemsForMediaRequest(request);
    }
    if (objectGraph.client.isiOS) {
        request.enablingFeature("categoryHeaders");
        request.enablingFeature("onDevicePersonalization");
    }
    // Enable category features
    if (objectGraph.bag.enableFeaturedCategoriesOnGroupings) {
        request.enablingFeature("featuredCategories");
    }
    // Enable category bricks
    if (objectGraph.bag.enableCategoryBricksOnGroupings) {
        request.enablingFeature("categoryBricks");
    }
}
//# sourceMappingURL=grouping-request.js.map