import * as serverData from "../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../foundation/media/attributes";
import * as mediaRelationship from "../../foundation/media/relationships";
import * as content from "../content/content";
import { recoMetricsDataForFCData } from "../grouping/shelf-controllers/grouping-shelf-controller-common";
import { collectionShelfDisplayStyleFromShelfData, extractEditorialClientParams } from "./editorial-data-util";
import { buildShelfMetricsOptions } from "./editorial-page-shelf-metrics";
import { GameCenterShelfClientFilter, } from "./editorial-page-types";
import { isSome } from "@jet/environment";
/**
 * Create the base shelf token to start parsing an editorial page shelf. This will create a new shelf token with the
 * base metrics options
 *
 * @param objectGraph The App Store dependency graph for making native calls and viewing properties
 * @param pageId The id of the editorial page this shelf is on.
 * @param shelfData The media api data for this specific shelf
 * @param isArcadePage Whether this shelf is on the arcade page or not
 * @param shelfIndex The index of this shelf on the page, at the time of building
 * @param metricsPageInformation The metrics page information for the editorial page
 * @param metricsLocationTracker The location tracker for building metrics location information
 * @param isSeeAll Whether the shelf is being built as a shelf for a sidepacked See All page.
 * @param collectionDisplayStyleOverride A way to override collection display style that is coming from mediaAPI.
 */
export function createBaseShelfToken(objectGraph, pageId, shelfData, isArcadePage, shelfIndex, metricsPageInformation, metricsLocationTracker, isSeeAll = false, collectionDisplayStyle = null) {
    const shelfToken = {
        id: serverData.asString(shelfData, "id"),
        type: shelfData.type,
        collectionDisplayStyle: collectionDisplayStyle !== null && collectionDisplayStyle !== void 0 ? collectionDisplayStyle : collectionShelfDisplayStyleFromShelfData(objectGraph, shelfData),
        filterOverrides: mediaAttributes.attributeAsArrayOrEmpty(shelfData, "filterOverrides"),
        pageId: pageId,
        data: shelfData,
        presentationHints: {},
        clientIdentifierOverride: null,
        isFirstRender: true,
        shouldFilter: true,
        gamesFilter: gamesFilterFromShelfData(shelfData),
        hasExistingContent: false,
        title: null,
        subtitle: null,
        eyebrow: null,
        titleArtwork: null,
        remainingItems: [],
        metricsImpressionOptions: null,
        metricsPageInformation: metricsPageInformation,
        metricsLocationTracker: metricsLocationTracker,
        recoMetricsData: recoMetricsDataForFCData(objectGraph, shelfData),
        isDeferring: false,
        showOrdinals: false,
        ordinalIndex: 1,
        isSearchLandingPage: false,
        isArcadePage: isArcadePage,
        shelfIndex: shelfIndex,
        isSeeAll: isSeeAll,
    };
    applyBaseShelfTokenTitleValues(objectGraph, shelfData, shelfToken);
    return shelfToken;
}
/**
 * Apply the values for the title, subtitle, and eyebrow to the shelf token, as well as the impressionOptions
 *
 * @param objectGraph The App Store dependency graph for making native calls and viewing properties
 * @param shelfData The media api data for this specific shelf
 * @param shelfToken The shelf token to apply the title values to
 */
export function applyBaseShelfTokenTitleValues(objectGraph, shelfData, shelfToken) {
    let title = mediaAttributes.attributeAsString(shelfData, "editorialNotes.name");
    let subtitle = mediaAttributes.attributeAsString(shelfData, "editorialNotes.tagline");
    let eyebrow = mediaAttributes.attributeAsString(shelfData, "editorialNotes.badge");
    let titleArtwork = null;
    // 'Similar To' Personalised Shelf
    // Check if personalised shelf has a badge-content, if it's got valid content, and the clients support eyebrows and title artwork:
    // - Reco shelf title becomes the eyebrow
    // - Featured App in badge-content becomes the shelf title, and we add the icon as title artwork
    // - No subtitle should be shown
    // - Flag to use the eyebrown name for metrics
    let wantsEyebrowNameForMetrics = false;
    const badgeContent = mediaRelationship.relationshipData(objectGraph, shelfData, "badge-content");
    if (serverData.isDefinedNonNullNonEmpty(badgeContent)) {
        eyebrow = objectGraph.loc.uppercased(mediaAttributes.attributeAsString(shelfData, "editorialNotes.name"));
        title = mediaAttributes.attributeAsString(badgeContent, "editorialNotes.name");
        titleArtwork = content.iconFromData(objectGraph, badgeContent, {
            useCase: 1 /* content.ArtworkUseCase.LockupIconSmall */,
        });
        subtitle = null;
        wantsEyebrowNameForMetrics = true;
    }
    if (preprocessor.GAMES_TARGET) {
        const editorialClientParams = extractEditorialClientParams(objectGraph, shelfData);
        // This flag hides the whole shelf header content altogether
        if (isSome(editorialClientParams) && editorialClientParams.suppressName) {
            title = null;
            subtitle = null;
            eyebrow = null;
            titleArtwork = null;
        }
    }
    const shelfMetricsOptionsTitle = wantsEyebrowNameForMetrics ? eyebrow : title;
    const shelfMetricsOptions = buildShelfMetricsOptions(objectGraph, shelfData, shelfMetricsOptionsTitle, shelfToken.metricsPageInformation, shelfToken.metricsLocationTracker);
    shelfToken.title = title;
    shelfToken.subtitle = subtitle;
    shelfToken.eyebrow = eyebrow;
    shelfToken.titleArtwork = titleArtwork;
    shelfToken.metricsImpressionOptions = shelfMetricsOptions;
}
function gamesFilterFromShelfData(shelfData) {
    const clientFilter = mediaAttributes.attributeAsString(shelfData, "clientFilter");
    let gamesFilter = null;
    switch (clientFilter) {
        case GameCenterShelfClientFilter.ArcadeGames:
            gamesFilter = "arcade";
            break;
        case GameCenterShelfClientFilter.AllGames:
            gamesFilter = "all";
            break;
        default:
            break;
    }
    return gamesFilter;
}
//# sourceMappingURL=editorial-page-shelf-token.js.map