import { attributeAsString } from "@apple-media-services/media-api";
import { isSome } from "@jet/environment/types/optional";
import * as models from "../../../api/models";
import { asBooleanOrFalse, isDefinedNonNullNonEmpty } from "../../../foundation/json-parsing/server-data";
import { attributeAsDictionary } from "../../../foundation/media/attributes";
import { metricsFromMediaApiObject } from "../../../foundation/media/data-structure";
import * as color from "../../../foundation/util/color-util";
import { shallowCopyOf } from "../../../foundation/util/objects";
import * as gamesComponentBuilder from "../../../gameservicesui/src/editorial-page/editorial-component-builder";
import * as content from "../../content/content";
import { deepLinkUrlFromData } from "../../linking/external-deep-link";
import { popLocation, pushBasicLocation } from "../../metrics/helpers/location";
import { areAppTagsEnabled } from "../../util/app-tags-util";
import { editorialDisplayOptionsFromClientParams, extractEditorialClientParams } from "../editorial-data-util";
import { editorialMediaDataForPlacement, EditorialMediaPlacement, isMediaDark, } from "../editorial-media-util";
import * as editorialComponentMediaUtil from "../editorial-page-component-media-util";
import { heroOverlayFromData } from "../editorial-page-hero-util";
import * as metricsHelpersImpressions from "../../metrics/helpers/impressions";
/**
 * Build a header shelf for the editorial page, if present.
 * @param objectGraph The App Store object graph.
 * @param pageData The data for the editorial page.
 * @param shelfToken The shelf token.
 * @returns A Shelf, if the data is available.
 */
export function buildHeaderShelf(objectGraph, pageData, shelfToken) {
    var _a, _b, _c;
    const shouldUseShelfData = isDefinedNonNullNonEmpty(attributeAsDictionary(shelfToken.data, "editorialNotes"));
    const headerData = shouldUseShelfData ? shelfToken.data : pageData;
    if (!headerData) {
        return null;
    }
    if (preprocessor.GAMES_TARGET) {
        return gamesComponentBuilder.makeGamesPageHeaderFromData(objectGraph, headerData, shelfToken);
    }
    const editorialArtworkData = isDefinedNonNullNonEmpty(attributeAsDictionary(headerData, "editorialArtwork"));
    if (editorialArtworkData) {
        const pageHeader = new models.HeroCarousel();
        const pageHeaderCarouselItem = new models.HeroCarouselItem();
        const editorialClientParams = extractEditorialClientParams(objectGraph, headerData);
        const pageHeaderCarouselItemOverlay = heroOverlayFromData(objectGraph, headerData, shelfToken, editorialClientParams);
        pageHeaderCarouselItemOverlay.overlayType = "text";
        pageHeaderCarouselItemOverlay.callToActionText = null;
        pageHeaderCarouselItem.clickAction = null;
        pageHeaderCarouselItemOverlay.collectionIcons = null;
        pageHeaderCarouselItemOverlay.lockup = null;
        pageHeaderCarouselItemOverlay.callToActionText = null;
        pageHeaderCarouselItem.overlay = pageHeaderCarouselItemOverlay;
        const lockupOptions = {
            artworkUseCase: 1 /* content.ArtworkUseCase.LockupIconSmall */,
            offerStyle: "transparent",
            offerEnvironment: "dark",
            canDisplayArcadeOfferButton: content.shelfDisplayStyleCanDisplayArcadeOfferButtons(objectGraph, shelfToken.collectionDisplayStyle),
            isContainedInPreorderExclusiveShelf: false,
            metricsOptions: {
                pageInformation: shelfToken.metricsPageInformation,
                locationTracker: shelfToken.metricsLocationTracker,
                recoMetricsData: metricsFromMediaApiObject(headerData),
            },
            externalDeepLinkUrl: deepLinkUrlFromData(objectGraph, headerData),
        };
        pushBasicLocation(objectGraph, lockupOptions.metricsOptions, "collectionIcons");
        const collectionIcons = editorialComponentMediaUtil.editorialFallbackAppIconsFromData(objectGraph, headerData, lockupOptions);
        popLocation(lockupOptions.metricsOptions.locationTracker);
        pageHeaderCarouselItem.collectionIcons = collectionIcons;
        const heroMediaData = editorialMediaDataForPlacement(objectGraph, headerData, EditorialMediaPlacement.Hero);
        const hideEditorialMedia = asBooleanOrFalse(editorialClientParams.hideEditorialMedia);
        populatePageHeaderCarouselItemMedia(objectGraph, pageHeaderCarouselItem, hideEditorialMedia, heroMediaData, false);
        pageHeaderCarouselItem.editorialDisplayOptions = editorialDisplayOptionsFromClientParams(editorialClientParams);
        pageHeader.items = [pageHeaderCarouselItem];
        const rtlPageHeaderCarouselItem = shallowCopyOf(pageHeaderCarouselItem);
        populatePageHeaderCarouselItemMedia(objectGraph, rtlPageHeaderCarouselItem, hideEditorialMedia, heroMediaData, true);
        pageHeader.rtlItems = [rtlPageHeaderCarouselItem];
        const shelf = new models.Shelf("heroCarousel");
        shelf.items = [pageHeader];
        return shelf;
    }
    else {
        const editorialName = attributeAsString(headerData, "name");
        if (isDefinedNonNullNonEmpty(editorialName) && areAppTagsEnabled(objectGraph, null)) {
            const shelf = new models.Shelf("mediaPageHeader");
            shelf.id = "mediaPageHeader";
            const editorialBadge = (_a = attributeAsString(headerData, "editorialNotes.badge")) !== null && _a !== void 0 ? _a : null;
            const lockupOptions = {
                artworkUseCase: 1 /* content.ArtworkUseCase.LockupIconSmall */,
                offerStyle: "transparent",
                offerEnvironment: "dark",
                canDisplayArcadeOfferButton: content.shelfDisplayStyleCanDisplayArcadeOfferButtons(objectGraph, shelfToken.collectionDisplayStyle),
                isContainedInPreorderExclusiveShelf: false,
                metricsOptions: {
                    pageInformation: shelfToken.metricsPageInformation,
                    locationTracker: shelfToken.metricsLocationTracker,
                    recoMetricsData: metricsFromMediaApiObject(headerData),
                },
                externalDeepLinkUrl: deepLinkUrlFromData(objectGraph, headerData),
                useJoeColorIconPlaceholder: true,
                joeColorPlaceholderSelectionLogic: content.bestJoeColorPlaceholderSelectionLogic,
            };
            let collectionIcons = (_b = editorialComponentMediaUtil.editorialFallbackAppIconsFromData(objectGraph, shelfToken.data, lockupOptions)) !== null && _b !== void 0 ? _b : undefined;
            let backgroundColor;
            if (isSome(collectionIcons) && (collectionIcons === null || collectionIcons === void 0 ? void 0 : collectionIcons.length) >= 3) {
                collectionIcons = collectionIcons.slice(0, 3);
                const collectionIconBackgroundColor = collectionIcons[0].backgroundColor;
                if (isSome(collectionIconBackgroundColor) && (collectionIconBackgroundColor === null || collectionIconBackgroundColor === void 0 ? void 0 : collectionIconBackgroundColor.type) === "rgb") {
                    backgroundColor = content.closestTagBackgroundColorForIcon(collectionIconBackgroundColor);
                }
                const isArtworkDark = color.isDarkColor(backgroundColor);
                const editorialClientParams = extractEditorialClientParams(objectGraph, shelfToken.data);
                const mediaPageHeader = new models.MediaPageHeader(editorialBadge, editorialName, null, undefined, undefined, collectionIcons, false, backgroundColor !== null && backgroundColor !== void 0 ? backgroundColor : undefined, convertHintToIconCollectionStyle((_c = editorialClientParams.fallbackDisplayStyleHint) !== null && _c !== void 0 ? _c : "one", collectionIcons === null || collectionIcons === void 0 ? void 0 : collectionIcons.length), isArtworkDark ? "dark" : "light");
                // Configure impressions
                const metricsOptions = {
                    pageInformation: shelfToken.metricsPageInformation,
                    locationTracker: shelfToken.metricsLocationTracker,
                    recoMetricsData: metricsFromMediaApiObject(headerData),
                };
                const impressionOptions = metricsHelpersImpressions.impressionOptionsForTagHeader(objectGraph, headerData, editorialName, metricsOptions);
                metricsHelpersImpressions.addImpressionFieldsToTagRoomHeader(objectGraph, mediaPageHeader, impressionOptions);
                shelf.items = [mediaPageHeader];
                return shelf;
            }
            else {
                return null;
            }
        }
        else {
            return null;
        }
    }
}
function convertHintToIconCollectionStyle(hint, artworkCount) {
    if (artworkCount === 2) {
        return "TwoUp";
    }
    else {
        switch (hint) {
            case "one":
                return "ThreeUp";
            case "two":
                return "Fan";
            case "three":
                return "Asymmetrical";
            default:
                return "ThreeUp";
        }
    }
}
function populatePageHeaderCarouselItemMedia(objectGraph, pageHeaderCarouselItem, hideEditorialMedia, heroMediaData, isRTL) {
    if (!hideEditorialMedia) {
        pageHeaderCarouselItem.backgroundColor = isRTL
            ? heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.rtlBackgroundColor
            : heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.backgroundColor;
        pageHeaderCarouselItem.artwork = isRTL ? heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.rtlArtwork : heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.artwork;
        pageHeaderCarouselItem.video = isRTL ? heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.rtlVideo : heroMediaData === null || heroMediaData === void 0 ? void 0 : heroMediaData.video;
    }
    pageHeaderCarouselItem.isMediaDark = isMediaDark(objectGraph, heroMediaData, isRTL);
}
//# sourceMappingURL=editorial-page-header-shelf-builder.js.map