// region Lockup Creation
import * as serverData from "../../foundation/json-parsing/server-data";
import * as mediaDataStructure from "../../foundation/media/data-structure";
import * as content from "../content/content";
import * as adLockups from "../lockups/ad-lockups";
import * as lockups from "../lockups/lockups";
import { clickTargetForCollectionDisplayStyle } from "./editorial-action-util";
import { CollectionShelfFilterOverride } from "./editorial-page-types";
import { isNothing } from "@jet/environment/types/optional";
// region Lockup Creation
/**
 * Create a lockup for shelfContents to display within an editorial page  shelf
 * @param objectGraph
 * @param lockupData shelfContents to create lockup for.
 * @param shelfToken Shelf shelfToken
 * @param itemPosition The position of the item in the shelf
 * @param collectionDisplayStyle The collection display style
 */
export function lockupFromData(objectGraph, lockupData, shelfToken, itemPosition, collectionDisplayStyle) {
    if (serverData.isNullOrEmpty(lockupData)) {
        return null;
    }
    // Set the ordinal
    let ordinalString;
    if (shelfToken.showOrdinals) {
        ordinalString = objectGraph.loc.decimal(shelfToken.ordinalIndex);
    }
    let offerStyle = null;
    if (serverData.isDefinedNonNull(shelfToken.shelfBackground) &&
        (shelfToken.shelfBackground.type === "color" || shelfToken.shelfBackground.type === "interactive")) {
        offerStyle = "white";
    }
    let clientIdentifierOverride;
    if (serverData.isDefinedNonNullNonEmpty(shelfToken)) {
        clientIdentifierOverride = shelfToken.clientIdentifierOverride;
    }
    const contentMetricsOptions = {
        ...shelfToken.metricsImpressionOptions,
        id: lockupData.id,
        idType: "its_id",
    };
    // Create the lockup
    const lockupOptions = {
        ordinal: ordinalString,
        metricsOptions: {
            ...contentMetricsOptions,
            recoMetricsData: mediaDataStructure.metricsFromMediaApiObject(lockupData),
            isAdvert: adLockups.isAdvert(objectGraph, lockupData),
            targetType: clickTargetForCollectionDisplayStyle(objectGraph, collectionDisplayStyle),
        },
        clientIdentifierOverride: clientIdentifierOverride,
        artworkUseCase: 0 /* content.ArtworkUseCase.Default */,
        offerStyle: offerStyle,
        canDisplayArcadeOfferButton: content.shelfDisplayStyleCanDisplayArcadeOfferButtons(objectGraph, collectionDisplayStyle),
        isContainedInPreorderExclusiveShelf: shelfToken.filterOverrides.includes(CollectionShelfFilterOverride.ShowOnlyPreorder),
    };
    let lockup;
    switch (collectionDisplayStyle) {
        case "EditorialLockupMediumVariant":
        case "EditorialLockupMedium":
        case "EditorialLockupLargeVariant":
        case "EditorialLockupLarge":
            lockupOptions.offerEnvironment = "light";
            lockup = lockups.imageLockupFromData(objectGraph, lockupData, lockupOptions, collectionDisplayStyle);
            break;
        case "Poster":
            lockup = lockups.posterLockupFromData(objectGraph, lockupData, lockupOptions);
            break;
        default:
            lockup = lockups.lockupFromData(objectGraph, lockupData, lockupOptions);
    }
    if (serverData.isNull(lockup) || !lockup.isValid()) {
        return null;
    }
    lockup.id = lockupIdentifier(shelfToken, itemPosition, lockup.adamId);
    return lockup;
}
/**
 * @param shelfToken The shelf token
 * @param itemPosition The position of the item in the shelf
 * @param adamId The adamID associated with the lockup
 * @returns A string for uniquely identifying the lockup in this shelf
 */
function lockupIdentifier(shelfToken, itemPosition, adamId) {
    if (isNothing(adamId)) {
        return null;
    }
    return `${shelfToken.id}_${itemPosition}_${adamId}`;
}
// endregion
//# sourceMappingURL=editorial-page-lockup-utils.js.map