import { Request, defaultAdditionalPlatformsForClient } from "../../foundation/media/data-fetching";
import { shouldFetchCustomAttributes } from "../product-page/product-page-variants";
import { shouldUsePrerenderedIconArtwork } from "../content/content";
/**
 * Returns the attributes to use for a developer media API request.
 */
export function developerAttributes(objectGraph) {
    const attributes = ["editorialArtwork", "editorialVideo", "requiredCapabilities", "minimumOSVersion"];
    if (objectGraph.client.isMac) {
        attributes.push("screenshotsByType");
    }
    else {
        attributes.push("isAppleWatchSupported");
    }
    if (objectGraph.appleSilicon.isSupportEnabled) {
        attributes.push("macRequiredCapabilities");
    }
    if (objectGraph.client.isMac) {
        attributes.push("hasMacIPAPackage");
    }
    if (objectGraph.client.isVision) {
        attributes.push("compatibilityControllerRequirement");
    }
    if (objectGraph.bag.enableUpdatedAgeRatings) {
        attributes.push("ageRating");
    }
    if (shouldUsePrerenderedIconArtwork(objectGraph)) {
        attributes.push("iconArtwork");
    }
    return attributes;
}
export const iosDeveloperRelationshipKeys = [
    "latest-release-app",
    "system-apps",
    "arcade-apps",
    "app-bundles",
    "ios-apps",
    "imessage-apps",
    "watch-apps",
    "atv-apps",
];
export const visionOSDeveloperRelationshipKeys = [
    "latest-release-app",
    "xros-apps",
    "arcade-apps",
    "ios-apps",
];
export const macosDeveloperRelationshipKeys = [
    "latest-release-app",
    "arcade-apps",
    "app-bundles",
    "mac-apps",
];
export const webDeveloperRelationshipKeys = [
    "latest-release-app",
    "system-apps",
    "arcade-apps",
    "app-bundles",
    "ios-apps",
    "imessage-apps",
    "watch-apps",
    "atv-apps",
    "xros-apps",
    "mac-apps",
];
export const watchosDeveloperRelationshipKey = "watch-apps";
function developerRelationships(objectGraph) {
    let relationships = [];
    switch (objectGraph.client.deviceType) {
        case "mac":
            relationships = relationships.concat(macosDeveloperRelationshipKeys);
            if (objectGraph.appleSilicon.isSupportEnabled) {
                relationships.push("mac-os-compatible-ios-apps");
            }
            break;
        case "watch":
            relationships.push(watchosDeveloperRelationshipKey);
            break;
        case "vision":
            relationships = relationships.concat(visionOSDeveloperRelationshipKeys);
            break;
        case "web":
            relationships = relationships.concat(webDeveloperRelationshipKeys);
            break;
        default:
            relationships = relationships.concat(iosDeveloperRelationshipKeys);
            break;
    }
    return relationships;
}
/**
 * Creates a {@linkcode Request} for a "developer" page with the given {@linkcode id}
 */
export function makeDeveloperRequest(objectGraph, id) {
    const request = new Request(objectGraph).withIdOfType(id, "developers");
    return addDeveloperRequestProperties(objectGraph, request);
}
/**
 * Add the expected request attributes (relationships, platforms, etc) to a request
 * for a "developer" resource
 */
export function addDeveloperRequestProperties(objectGraph, request) {
    request
        .includingAdditionalPlatforms(defaultAdditionalPlatformsForClient(objectGraph))
        .includingRelationships(developerRelationships(objectGraph))
        .includingAttributes(developerAttributes(objectGraph))
        .includingMacOSCompatibleIOSAppsWhenSupported()
        .usingCustomAttributes(shouldFetchCustomAttributes(objectGraph));
    if (objectGraph.client.isWeb) {
        // The "web" client needs to load *all* of the data for SEO purposes
        request.addingQuery("sparseLimit[developers:ios-apps]", "40");
    }
    return request;
}
//# sourceMappingURL=developer-request.js.map