/**
 * Build methods for Arcade See All Games Facets.
 */
import * as models from "../../api/models";
import { isDefinedNonNull, isDefinedNonNullNonEmpty, isNullOrEmpty } from "../../foundation/json-parsing/server-data";
import { Parameters } from "../../foundation/network/url-constants";
import * as metricsClickHelpers from "../metrics/helpers/clicks";
// All Facet Options
export const sortReleaseDateFacetOptionValue = `-releaseDateByDay&sort=name`;
export const sortLastUpdatedFacetOptionValue = `-lastUpdatedByDay&sort=name`;
export const sortNameFacetOptionValue = `alphabet&sort=name`;
export const sortCategoryFacetOptionValue = `category&sort=-releaseDate`;
export const sortFacetOptionParameter = "groupBy";
export const comingSoonFacetOptionParameter = "with";
export const comingSoonAppsFacetOptionValue = "comingSoonApps";
export const comingSoonGroupingFacetOptionValue = "comingSoonGrouping";
/**
 * The facet for filtering bincompat games in visionOS
 * By default, not passing this parameter will includes both bincompat and native games.
 * To filter native games only, passing `filter[platform]=reality`
 */
export const binCompatGamesFacetOptionParameter = "filter[platform]";
// The facet option for toggling video preview in lockups.
// This facet is used for toggle `isCompactMode` locally, it don't match with any media api parameter.
export const gamePreviewsFacetOptionParameter = "gamePreviews";
export const facetOptionsParameterMapping = {};
let areFacetsInitialized = false;
let sortReleaseDateFacetOption = null;
let sortLastUpdatedFacetOption = null;
let sortNameFacetOption = null;
let sortCategoryFacetOption = null;
let ageRatingDefaultFacetOption = null;
let ageRating4PlusFacetOption = null;
let ageRating9PlusFacetOption = null;
let ageRating12PlusFacetOption = null;
let controllerSupportFacetOption = null;
let multiplayerSupportFacetOption = null;
let comingSoonFacetOption = null;
let includeBinCompatGamesFacetOption = null;
let onlyNativeGamesFacetOption = null;
let gamePreviewsFacetOption = null;
// Mapping of external facet parameters to media api paramter values
const facetParameterMediaApiMapping = {};
facetParameterMediaApiMapping[Parameters.sort] = sortFacetOptionParameter;
facetParameterMediaApiMapping[Parameters.ageRating] = "filter[ageRating]";
facetParameterMediaApiMapping[Parameters.controllerSupport] = "filter[supportsGameController]";
facetParameterMediaApiMapping[Parameters.multiplayerSupport] = "filter[isMultiplayer]";
facetParameterMediaApiMapping[Parameters.comingSoon] = comingSoonFacetOptionParameter;
facetParameterMediaApiMapping[Parameters.binCompatGames] = binCompatGamesFacetOptionParameter;
function initializeFacets(objectGraph) {
    if (areFacetsInitialized) {
        return;
    }
    areFacetsInitialized = true;
    sortReleaseDateFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_SORTS_RELEASE_DATE"), sortReleaseDateFacetOptionValue, "releaseDate");
    sortLastUpdatedFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_SORTS_LAST_UPDATED"), sortLastUpdatedFacetOptionValue, "lastUpdated");
    sortNameFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_SORTS_NAME"), sortNameFacetOptionValue, "name");
    sortCategoryFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_SORTS_CATEGORY"), sortCategoryFacetOptionValue, "category");
    ageRatingDefaultFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_DEFAULT_ALL"), null);
    ageRating4PlusFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_FILTERS_AGE_VALUE_4PLUS"), "4Plus");
    ageRating9PlusFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_FILTERS_AGE_VALUE_9PLUS"), "9Plus");
    ageRating12PlusFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_FILTERS_AGE_VALUE_12PLUS"), "12Plus");
    controllerSupportFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_FILTERS_CONTROLLER_SUPPORT"), "true", null, "gamecontroller");
    multiplayerSupportFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_FILTERS_MULTIPLAYER"), "true", null, "person.2");
    comingSoonFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_FILTERS_COMING_SOON"), "true");
    if (objectGraph.client.isVision) {
        includeBinCompatGamesFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_FILTERS_BIN_COMPAT_GAMES"), null);
        onlyNativeGamesFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_FILTERS_BIN_COMPAT_GAMES"), "realityDevice");
        gamePreviewsFacetOption = new models.PageFacetOption(objectGraph.loc.string("PAGE_FACETS_FILTERS_GAME_PREVIEWS"), null);
    }
    facetOptionsParameterMapping[Parameters.sort] = {
        releaseDate: [sortReleaseDateFacetOption],
        lastUpdated: [sortLastUpdatedFacetOption],
        name: [sortNameFacetOption],
        category: [sortCategoryFacetOption],
    };
    facetOptionsParameterMapping[Parameters.ageRating] = {
        "4plus": [ageRating4PlusFacetOption],
        "9plus": [ageRating9PlusFacetOption],
        "12plus": [ageRating12PlusFacetOption],
    };
    facetOptionsParameterMapping[Parameters.controllerSupport] = {
        true: [controllerSupportFacetOption],
        false: [],
    };
    facetOptionsParameterMapping[Parameters.multiplayerSupport] = {
        true: [multiplayerSupportFacetOption],
        false: [],
    };
    facetOptionsParameterMapping[Parameters.comingSoon] = {
        true: [comingSoonFacetOption],
        false: [],
    };
    if (objectGraph.client.isVision) {
        facetOptionsParameterMapping[Parameters.binCompatGames] = {
            true: [includeBinCompatGamesFacetOption],
            false: [onlyNativeGamesFacetOption],
        };
        facetOptionsParameterMapping[Parameters.gamePreviews] = {
            true: [gamePreviewsFacetOption],
            false: [],
        };
    }
}
/**
 *
 * Create Page Facets. Theres platform specific variations here.
 */
export function createArcadeSeeAllGamesFacets(objectGraph) {
    initializeFacets(objectGraph);
    const isEnabledVisionSAGSegmentedNavFeature = objectGraph.featureFlags.isEnabled("see_all_games_segmented_nav_2024A");
    const ageRatingsFacetTitle = objectGraph.client.isMac
        ? objectGraph.loc.string("PAGE_FACETS_AGE_RATINGS_TITLE")
        : objectGraph.loc.string("PAGE_FACETS_FILTERS_AGE_RATINGS");
    const ageRatingsFacet = new models.PageFacetsFacet(Parameters.ageRating, "filter[ageRating]", ageRatingsFacetTitle, "singleSelection", [ageRatingDefaultFacetOption, ageRating4PlusFacetOption, ageRating9PlusFacetOption, ageRating12PlusFacetOption], [ageRatingDefaultFacetOption], "age", pageFacetChangeAction(objectGraph, "sort"));
    // Sorts
    const isEnabledArcadeSeeAllGamesUplift = objectGraph.featureFlags.isEnabled("arcade_see_all_games_menu_uplift");
    const isFeatureFlagEnabled = (objectGraph.client.isiOS && isEnabledArcadeSeeAllGamesUplift) ||
        (objectGraph.client.isVision && isEnabledVisionSAGSegmentedNavFeature);
    const sortsFacet = new models.PageFacetsFacet(Parameters.sort, sortFacetOptionParameter, objectGraph.loc.string("PAGE_FACETS_SORT_BY_TITLE"), "singleSelection", [sortReleaseDateFacetOption, sortLastUpdatedFacetOption, sortNameFacetOption, sortCategoryFacetOption], isFeatureFlagEnabled ? [sortReleaseDateFacetOption] : null, "sort", pageFacetChangeAction(objectGraph, "sort"));
    // Controllers
    const controllerSupportFacetOptionFacetTitle = objectGraph.client.isMac
        ? objectGraph.loc.string("PAGE_FACETS_FILTERS_CONTROLLERS")
        : objectGraph.loc.string("PAGE_FACETS_FILTERS_CONTROLLER_SUPPORT");
    const controllerSupportFacetOptionFacet = new models.PageFacetsFacet(Parameters.controllerSupport, "filter[supportsGameController]", controllerSupportFacetOptionFacetTitle, "toggle", [controllerSupportFacetOption], null, "controllerSupport", pageFacetChangeAction(objectGraph, "sort"));
    const multiplayerFacetTitle = objectGraph.client.isMac
        ? objectGraph.loc.string("PAGE_FACETS_FILTERS_MULTIPLAYER")
        : objectGraph.loc.string("PAGE_FACETS_FILTERS_MULTIPLAYER_GAMES");
    const multiplayerSupportFacetOptionFacet = new models.PageFacetsFacet(Parameters.multiplayerSupport, "filter[isMultiplayer]", multiplayerFacetTitle, "toggle", [multiplayerSupportFacetOption], null, "multiplayer", pageFacetChangeAction(objectGraph, "sort"));
    const comingSoonFacet = new models.PageFacetsFacet(Parameters.comingSoon, comingSoonFacetOptionParameter, objectGraph.loc.string("PAGE_FACETS_FILTERS_COMING_SOON"), "toggle", [comingSoonFacetOption], [comingSoonFacetOption], "comingSoon", pageFacetChangeAction(objectGraph, "comingSoon"));
    comingSoonFacet.isHiddenFromMenu = isFeatureFlagEnabled;
    const pageFacets = new models.PageFacets([], false, null);
    if (objectGraph.client.isVision) {
        sortsFacet.displayOptionsInline = true;
        pageFacets.facetGroups.push(new models.PageFacetsGroup([sortsFacet], objectGraph.loc.string("PAGE_FACETS_SORT_BY_TITLE_LOWERCASE")));
        ageRatingsFacet.showsSelectedOptions = true;
        pageFacets.facetGroups.push(new models.PageFacetsGroup([controllerSupportFacetOptionFacet, multiplayerSupportFacetOptionFacet, ageRatingsFacet], objectGraph.loc.string("PAGE_FACETS_FILTER_BY_TITLE_LOWERCASE")));
        const binCompatGamesFacet = createBinCompatGamesFacet(objectGraph);
        const gamePreviewsFacet = new models.PageFacetsFacet(Parameters.gamePreviews, gamePreviewsFacetOptionParameter, objectGraph.loc.string("PAGE_FACETS_FILTERS_GAME_PREVIEWS"), "toggle", [gamePreviewsFacetOption], [gamePreviewsFacetOption], "gamePreviews", pageFacetChangeAction(objectGraph, "gamePreviews"));
        const includeGroupsTitle = objectGraph.client.isVision && isEnabledVisionSAGSegmentedNavFeature
            ? objectGraph.loc.string("PAGE_FACETS_SHOW_TITLE_LOWERCASE")
            : objectGraph.loc.string("PAGE_FACETS_INCLUDE_TITLE_LOWERCASE");
        pageFacets.facetGroups.push(new models.PageFacetsGroup([binCompatGamesFacet, gamePreviewsFacet, comingSoonFacet], includeGroupsTitle));
    }
    else {
        switch (objectGraph.client.deviceType) {
            case "mac":
                pageFacets.facetGroups.push(new models.PageFacetsGroup([ageRatingsFacet]));
                if (objectGraph.bag.enableComingSoonToggle) {
                    pageFacets.facetGroups.push(new models.PageFacetsGroup([comingSoonFacet], objectGraph.loc.string("PAGE_FACETS_INCLUDE_TITLE")));
                }
                pageFacets.facetGroups.push(new models.PageFacetsGroup([controllerSupportFacetOptionFacet, multiplayerSupportFacetOptionFacet], objectGraph.loc.string("PAGE_FACETS_SUPPORTS_TITLE")));
                pageFacets.facetGroups.push(new models.PageFacetsGroup([sortsFacet]));
                break;
            case "tv":
                pageFacets.facetGroups.push(new models.PageFacetsGroup([ageRatingsFacet], objectGraph.loc.string("PAGE_FACETS_AGE_RATINGS_TITLE_LOWERCASE")));
                const filterGroupFacets = [
                    controllerSupportFacetOptionFacet,
                    multiplayerSupportFacetOptionFacet,
                ];
                if (objectGraph.bag.enableComingSoonToggle) {
                    filterGroupFacets.push(comingSoonFacet);
                }
                pageFacets.facetGroups.push(new models.PageFacetsGroup(filterGroupFacets, objectGraph.loc.string("PAGE_FACETS_FILTERS_TITLE")));
                pageFacets.facetGroups.push(new models.PageFacetsGroup([sortsFacet], objectGraph.loc.string("PAGE_FACETS_SORT_BY_TITLE_LOWERCASE")));
                break;
            default:
                sortsFacet.displayOptionsInline = true;
                pageFacets.facetGroups.push(new models.PageFacetsGroup([sortsFacet], objectGraph.loc.string("PAGE_FACETS_SORT_BY_TITLE")));
                ageRatingsFacet.showsSelectedOptions = true;
                pageFacets.facetGroups.push(new models.PageFacetsGroup([controllerSupportFacetOptionFacet, multiplayerSupportFacetOptionFacet, ageRatingsFacet], objectGraph.loc.string("PAGE_FACETS_FILTER_BY_TITLE")));
                if (objectGraph.bag.enableComingSoonToggle) {
                    pageFacets.facetGroups.push(new models.PageFacetsGroup([comingSoonFacet], objectGraph.loc.string("PAGE_FACETS_INCLUDE_TITLE")));
                }
                break;
        }
    }
    return pageFacets;
}
function createBinCompatGamesFacet(objectGraph) {
    const binCompatGamesFacet = new models.PageFacetsFacet(Parameters.binCompatGames, binCompatGamesFacetOptionParameter, objectGraph.loc.string("PAGE_FACETS_FILTERS_BIN_COMPAT_GAMES"), "toggle", [includeBinCompatGamesFacetOption], [includeBinCompatGamesFacetOption], "binCompatGames", pageFacetChangeAction(objectGraph, "binCompatGames"));
    // Hiding this facet from the Filters menu, because this facet will be control by the page segmented picker.
    binCompatGamesFacet.isHiddenFromMenu = objectGraph.featureFlags.isEnabled("see_all_games_segmented_nav_2024A");
    return binCompatGamesFacet;
}
export function createDefaultSelectedFacetOptions(objectGraph, urlParameters = {}) {
    initializeFacets(objectGraph);
    const selectedFacetOptions = {
        "filter[ageRating]": [ageRatingDefaultFacetOption],
    };
    selectedFacetOptions[sortFacetOptionParameter] = [sortReleaseDateFacetOption];
    selectedFacetOptions[comingSoonFacetOptionParameter] = [comingSoonFacetOption];
    if (objectGraph.client.isVision) {
        selectedFacetOptions[binCompatGamesFacetOptionParameter] = [includeBinCompatGamesFacetOption];
        selectedFacetOptions[gamePreviewsFacetOptionParameter] = [gamePreviewsFacetOption];
    }
    const availableParameters = [
        Parameters.sort,
        Parameters.ageRating,
        Parameters.controllerSupport,
        Parameters.multiplayerSupport,
        Parameters.comingSoon,
    ];
    if (objectGraph.client.isVision) {
        availableParameters.push(Parameters.binCompatGames);
    }
    for (const parameter of availableParameters) {
        const facetOption = urlParameters[parameter];
        if (isNullOrEmpty(facetOption)) {
            continue;
        }
        const selectedFacetOption = facetOptionsParameterMapping[parameter][facetOption];
        const mediaApiFacetName = facetParameterMediaApiMapping[parameter];
        if (isDefinedNonNull(selectedFacetOption) && isDefinedNonNullNonEmpty(mediaApiFacetName)) {
            selectedFacetOptions[`${mediaApiFacetName}`] = selectedFacetOption;
        }
    }
    return selectedFacetOptions;
}
function pageFacetChangeAction(objectGraph, facetParameter) {
    const action = new models.BlankAction();
    metricsClickHelpers.addClickEventToPageFacetsChangeAction(objectGraph, action, facetParameter);
    return action;
}
export function createPageSegments(objectGraph) {
    const isSegmentedNavEnabled = objectGraph.client.isVision && objectGraph.featureFlags.isEnabled("see_all_games_segmented_nav_2024A");
    if (!isSegmentedNavEnabled) {
        return [];
    }
    else {
        const binCompatGamesFacet = createBinCompatGamesFacet(objectGraph);
        const allGamesPageSegment = {
            id: "all_games",
            title: objectGraph.loc.string("Arcade.SeeAllGames.PageSegment.AllGames.Title"),
            segmentAction: new models.ArcadeSeeAllGamesPageSegmentChangeAction(binCompatGamesFacet, includeBinCompatGamesFacetOption),
        };
        const visionGamesOnlyPageSegment = {
            id: "vision",
            title: objectGraph.loc.string("Arcade.SeeAllGames.PageSegment.AppleVisionGames.Title"),
            segmentAction: new models.ArcadeSeeAllGamesPageSegmentChangeAction(binCompatGamesFacet, undefined),
        };
        return [allGamesPageSegment, visionGamesOnlyPageSegment];
    }
}
//# sourceMappingURL=arcade-see-all-games-facets.js.map