import { isNothing, isSome } from "@jet/environment/types/optional";
import * as models from "../../api/models";
import * as serverData from "../../foundation/json-parsing/server-data";
import * as mediaRelationship from "../../foundation/media/relationships";
import * as mediaAttributes from "../../foundation/media/attributes";
import * as objects from "../../foundation/util/objects";
import * as metricsBuilder from "../metrics/builder";
import * as metricsHelpersLocation from "../metrics/helpers/location";
import * as metricsHelpersPage from "../metrics/helpers/page";
import { MetricsReferralContext } from "../metrics/metrics-referral-context";
import * as appPromotionCommon from "./app-promotions-common";
/**
 * Create a flow action for navigating to the offer detail page.
 * @param data The data blob
 * @param parentAppData The associated parent app data
 * @param offerItem The source offer object
 * @param baseMetricsOptions The base metrics options
 * @param animationBehavior The animation behavior for presenting the modal page
 * @param includeLockupClickAction Whether to generate a click action for the lockup
 * @param referrerData Referrer data from an incoming deep link
 */
export function offerItemDetailPageFlowActionFromData(objectGraph, data, parentAppData, offerItem, baseMetricsOptions, animationBehavior, includeLockupClickAction, referrerData) {
    const page = offerItemDetailPageFromData(objectGraph, data, parentAppData, offerItem, baseMetricsOptions, includeLockupClickAction, referrerData, false);
    const action = new models.FlowAction("offerItemDetail");
    action.title = offerItem.title;
    action.pageData = page;
    action.animationBehavior = animationBehavior;
    if (baseMetricsOptions && baseMetricsOptions.pageInformation) {
        action.referrerUrl = baseMetricsOptions.pageInformation.pageUrl;
    }
    return action;
}
/**
 * Creates an offer item detail page
 * @param objectGraph The object graph
 * @param data The data blob
 * @param parentAppData The data blob for the related parent app
 * @param offerItem The source offer
 * @param baseMetricsOptions The base metrics options to use for the detail page
 * @param includeLockupClickAction Whether to generate a click action for the lockup
 * @param referrerData The referrer data
 * @param isArcadePage Whether or not this is presented on the Arcade page
 */
export function offerItemDetailPageFromData(objectGraph, offerItemData, parentAppData, offerItem, baseMetricsOptions, includeLockupClickAction, referrerData, isArcadePage) {
    var _a, _b;
    let artwork = appPromotionCommon.artworkFromPlatformData(objectGraph, offerItemData, "productArtwork");
    // rdar://126775681 (Remove  force moduleArtwork to be null as MAPI still sends down artwork when it should not)
    // The below `if` check should be uncommented when this is removed - it was commented as strict null checking is smart enough to know that `artwork` will always be null.
    artwork = null;
    const mediaOverlayStyle = "dark";
    const isArtworkDark = true;
    const includeBorderInDarkMode = false;
    // if (serverData.isDefinedNonNullNonEmpty(artwork)) {
    //     isArtworkDark = color.isDarkColor(artwork?.backgroundColor);
    //     mediaOverlayStyle = isArtworkDark ? "dark" : "light";
    //     includeBorderInDarkMode = color.isDarkColor(artwork?.backgroundColor, 10);
    // }
    const copy = objects.shallowCopyOf(offerItem);
    const pageInformation = metricsHelpersPage.pageInformationForAppPromotionDetailPage(objectGraph, models.AppPromotionType.OfferItem, offerItemData.id, parentAppData.id, referrerData, (_a = baseMetricsOptions.recoMetricsData) !== null && _a !== void 0 ? _a : null);
    const metricsOptions = {
        ...baseMetricsOptions,
        pageInformation: pageInformation,
        locationTracker: metricsHelpersLocation.newLocationTracker(),
    };
    const offerEnvironment = isArtworkDark ? "dark" : "light";
    const lockup = appPromotionCommon.lockupFromData(objectGraph, offerItemData, parentAppData, (_b = copy.title) !== null && _b !== void 0 ? _b : undefined, offerEnvironment, "transparent", false, metricsOptions, includeLockupClickAction, referrerData, isArcadePage, false);
    if (serverData.isNull(lockup)) {
        return null;
    }
    copy.offerLockup = lockup;
    const additionalInfoLabel = objectGraph.loc.string("Winback.AdditionalInfoButton.Title");
    const iapData = mediaRelationship.relationshipData(objectGraph, offerItemData, "salables");
    const redemptionExpirationDate = mediaAttributes.attributeAsString(offerItemData, "redemptionExpirationDate");
    let additionalInfo;
    if (isSome(redemptionExpirationDate) && isSome(iapData)) {
        const endDate = new Date(redemptionExpirationDate);
        additionalInfo = formattedAdditionalInfoFromData(objectGraph, offerItemData, iapData, endDate);
    }
    const page = new models.OfferItemDetailPage(copy, artwork !== null && artwork !== void 0 ? artwork : undefined, undefined, mediaOverlayStyle, includeBorderInDarkMode, additionalInfoLabel, additionalInfo);
    page.backButtonActionMetrics.addMetricsData(createButtonActionMetricsForTarget(objectGraph, "back", pageInformation, metricsOptions.locationTracker));
    page.learnMoreActionMetrics.addMetricsData(createButtonActionMetricsForTarget(objectGraph, "LearnMore", pageInformation, metricsOptions.locationTracker, additionalInfoLabel));
    page.closeButtonActionMetrics.addMetricsData(createButtonActionMetricsForTarget(objectGraph, "close", pageInformation, metricsOptions.locationTracker));
    metricsHelpersPage.addMetricsEventsToPageWithInformation(objectGraph, page, pageInformation, (fields) => {
        if (serverData.isDefinedNonNullNonEmpty(referrerData)) {
            MetricsReferralContext.shared.addReferralContextToMetricsFieldsIfNecessary(fields);
        }
    });
    return page;
}
function createButtonActionMetricsForTarget(objectGraph, targetId, pageInformation, locationTracker, title) {
    let actionType;
    switch (targetId) {
        case "LearnMore":
            actionType = "navigate";
            break;
        case "back":
            actionType = "back";
            break;
        case "close":
            actionType = "dismiss";
            break;
        default:
            break;
    }
    const eventFields = {
        targetType: "button",
        actionType,
        targetId,
        idType: undefined,
        location: metricsHelpersLocation.createContentLocation(objectGraph, {
            pageInformation: pageInformation,
            locationTracker: locationTracker,
            targetType: "button",
            id: targetId,
        }, title !== null && title !== void 0 ? title : targetId),
    };
    const event = metricsBuilder.createMetricsClickData(objectGraph, targetId, "button", eventFields);
    return event;
}
/**
 * Creates the additional info text of the promotion out of a templated string
 * @param data A data blob
 * @param iapData A data blob containing the iap offer
 * @returns Creates a formatted paragraph containing the contents of the additional info page.
 */
function formattedAdditionalInfoFromData(objectGraph, data, iapData, endDate) {
    if (isNothing(iapData)) {
        return undefined;
    }
    const skuName = mediaAttributes.attributeAsString(iapData, "name");
    const skuNameFormatted = isSome(skuName) ? "<b>" + skuName + "</b>" : undefined;
    const skuDescription = mediaAttributes.attributeAsString(iapData, "description.standard");
    const skuDescriptionFormatted = isSome(skuDescription) ? skuDescription + "<br>" : undefined;
    const termsTitle = "<b>" + objectGraph.loc.string("Promotion.Terms.Title") + "</b>";
    let terms;
    const redemptionDateFormat = objectGraph.loc.string("OfferItems.FormattedDate.RedemptionDate.DateFormat");
    if (isSome(skuName) && isSome(endDate)) {
        const templateKeyMap = {
            "@@redemptionDate@@": objectGraph.loc.formatDate(redemptionDateFormat, endDate),
            "@@skuName@@": skuName,
        };
        terms = appPromotionCommon.replacingTemplatedKeys(mediaAttributes.attributeAsString(data, "additionalTerms"), templateKeyMap);
    }
    const combinedStrings = [skuNameFormatted, skuDescriptionFormatted, termsTitle, terms].filter(isSome).join("<br>");
    return new models.Paragraph(combinedStrings, "text/x-apple-as3-nqml");
}
//# sourceMappingURL=offer-item-detail.js.map