import * as models from "../../api/models";
import * as metricsHelpersLocation from "../metrics/helpers/location";
import * as metricsHelpersPage from "../metrics/helpers/page";
import * as serverData from "../../foundation/json-parsing/server-data";
import * as mediaAttributes from "../../foundation/media/attributes";
import * as color from "../../foundation/util/color-util";
import * as objects from "../../foundation/util/objects";
import { addClickEventToAction } from "../metrics/helpers/clicks";
import { MetricsReferralContext } from "../metrics/metrics-referral-context";
import * as sharing from "../sharing";
import * as appPromotionCommon from "./app-promotions-common";
import { isNothing } from "@jet/environment";
import { makeAppEventPageIntent } from "../../api/intents/app-event-page-intent";
import { getLocale } from "../locale";
import { getPlatform } from "../preview-platform";
/**
 * Create a flow action for navigating to the app event detail page.
 * @param data The data blob
 * @param parentAppData The associated parent app data
 * @param appEvent The source app event
 * @param baseMetricsOptions The base metrics options
 * @param animationBehavior The animation behaviour for presenting the modal page
 * @param includeLockupClickAction Whether to generate a click action for the lockup
 * @param referrerData Referrer data from an incoming deep link
 */
export function appEventDetailPageFlowActionFromData(objectGraph, data, parentAppData, appEvent, baseMetricsOptions, animationBehavior, includeLockupClickAction, referrerData) {
    const page = appEventDetailPageFromData(objectGraph, data, parentAppData, appEvent, baseMetricsOptions, includeLockupClickAction, referrerData, false);
    const action = new models.FlowAction("appEventDetail");
    action.title = appEvent.title;
    action.pageData = page;
    action.animationBehavior = animationBehavior;
    if (baseMetricsOptions && baseMetricsOptions.pageInformation) {
        action.referrerUrl = baseMetricsOptions.pageInformation.pageUrl;
    }
    if (objectGraph.client.isWeb) {
        action.destination = makeAppEventPageIntent({
            ...getLocale(objectGraph),
            ...getPlatform(objectGraph),
            id: appEvent.appEventId,
        });
        action.pageUrl = mediaAttributes.attributeAsString(data, "url");
    }
    return action;
}
/**
 * Creates an app event detail page
 * @param objectGraph The object graph
 * @param data The data blob
 * @param parentAppData The data blob for the related parent app
 * @param appEvent The source app event
 * @param baseMetricsOptions The base metrics options to use for the detail page
 * @param includeLockupClickAction Whether to generate a click action for the lockup
 * @param referrerData The referrer data
 * @param isArcadePage Whether or not this is presented on the Arcade page
 */
export function appEventDetailPageFromData(objectGraph, data, parentAppData, appEvent, baseMetricsOptions, includeLockupClickAction, referrerData, isArcadePage) {
    var _a, _b;
    const artwork = appPromotionCommon.artworkFromData(objectGraph, data, "productArtwork");
    const video = appPromotionCommon.videoFromData(objectGraph, data, "productVideo", true, true);
    const copy = objects.shallowCopyOf(appEvent);
    const selectedArtwork = (_a = video === null || video === void 0 ? void 0 : video.preview) !== null && _a !== void 0 ? _a : artwork;
    let mediaOverlayStyle = "dark";
    let isArtworkDark = true;
    let includeBorderInDarkMode = false;
    if (serverData.isDefinedNonNull(selectedArtwork)) {
        isArtworkDark = color.isDarkColor(selectedArtwork.backgroundColor);
        includeBorderInDarkMode = color.isDarkColor(selectedArtwork.backgroundColor, 10);
        mediaOverlayStyle = isArtworkDark ? "dark" : "light";
    }
    const pageInformation = metricsHelpersPage.pageInformationForAppPromotionDetailPage(objectGraph, models.AppPromotionType.AppEvent, data.id, parentAppData.id, referrerData, (_b = baseMetricsOptions.recoMetricsData) !== null && _b !== void 0 ? _b : null);
    const metricsOptions = {
        ...baseMetricsOptions,
        pageInformation: pageInformation,
        locationTracker: metricsHelpersLocation.newLocationTracker(),
        targetType: "EventDetails",
    };
    const clickOptions = {
        ...metricsOptions,
        id: parentAppData.id,
        inAppEventId: data.id,
        relatedSubjectIds: [parentAppData.id],
    };
    copy.notificationConfig = appPromotionCommon.notificationConfigFromData(objectGraph, data, appEvent, clickOptions, false);
    const shareAction = shareActionFromData(objectGraph, data, appEvent, clickOptions);
    const offerEnvironment = isArtworkDark ? "dark" : "light";
    const lockup = appPromotionCommon.lockupFromData(objectGraph, data, parentAppData, copy.title, offerEnvironment, "transparent", false, metricsOptions, includeLockupClickAction, referrerData, isArcadePage, false);
    if (serverData.isNull(lockup)) {
        return null;
    }
    copy.lockup = lockup;
    const page = new models.AppEventDetailPage(copy, artwork, video, shareAction, mediaOverlayStyle, includeBorderInDarkMode);
    metricsHelpersPage.addMetricsEventsToPageWithInformation(objectGraph, page, pageInformation, (fields) => {
        if (serverData.isDefinedNonNullNonEmpty(referrerData)) {
            MetricsReferralContext.shared.addReferralContextToMetricsFieldsIfNecessary(fields);
        }
    });
    return page;
}
/**
 * Creates an action for sharing an app event
 * @param data: The data blob
 * @param appEvent The source app event
 * @param metricsClickOptions The click options used for the containing context.
 */
function shareActionFromData(objectGraph, data, appEvent, metricsClickOptions) {
    var _a, _b;
    const url = mediaAttributes.attributeAsString(data, "url");
    if (isNothing(url) || url.length === 0) {
        return null;
    }
    // Prefer the module artwork first, otherwise fall back to the video preview frame, if available.
    const artwork = (_a = appEvent.moduleArtwork) !== null && _a !== void 0 ? _a : (_b = appEvent.moduleVideo) === null || _b === void 0 ? void 0 : _b.preview;
    let subtitle = objectGraph.loc.string("SHARE_APP_EVENT_SUBTITLE");
    if (subtitle === "SHARE_APP_EVENT_SUBTITLE") {
        subtitle = appEvent.subtitle;
    }
    const shareData = sharing.shareSheetDataForAppEvent(objectGraph, appEvent.title, subtitle, url, undefined, artwork);
    if (!serverData.isDefinedNonNull(shareData)) {
        return null;
    }
    const activities = sharing.shareSheetActivitiesForAppEvent(objectGraph, appEvent, url);
    const action = new models.ShareSheetAction(shareData, activities);
    addClickEventToAction(objectGraph, action, {
        ...metricsClickOptions,
        targetType: "lockup",
        actionType: "share",
        idType: "its_id",
    });
    return action;
}
//# sourceMappingURL=app-event-detail.js.map