import { GenericPage, AppEventDetailShelf, Shelf, Banner } from "../../api/models";
import { attributeAsString } from "../../foundation/media/attributes";
import { defaultAdditionalPlatformsForClient, Request } from "../../foundation/media/data-fetching";
import { dataFromDataContainer } from "../../foundation/media/data-structure";
import { relationshipData } from "../../foundation/media/relationships";
import { appEventOrPromotionStartDateFromData } from "../app-promotions/app-event";
import { appEventDetailPageFromData } from "../app-promotions/app-event-detail";
import { supportedAppPlatformsFromData } from "../content/content";
import { newLocationTracker } from "../metrics/helpers/location";
import { metricsPageInformationFromMediaApiResponse } from "../metrics/helpers/page";
import { create as createBanner } from "../product-page/banner";
/**
 * Generates a {@linkcode Request} to the `app-events` Media API endpoint
 */
export function makeAppEventPageRequest(objectGraph, intent) {
    const mediaApiRequest = new Request(objectGraph)
        .withIdOfType(intent.id, "app-events")
        .includingAdditionalPlatforms(defaultAdditionalPlatformsForClient(objectGraph))
        .includingRelationships(["app"]);
    if (objectGraph.client.isWeb) {
        mediaApiRequest.includingScopedAttributes("app-events", ["description", "productArtwork", "productVideo"]);
        mediaApiRequest.includingScopedAvailableIn("app-events", ["past"]);
    }
    return mediaApiRequest;
}
/**
 * Builds a `GenericPage` that represents an App Event detail page
 *
 * This is used by the "web" client to build the view-model for the standalone
 * "app event" pages
 */
export function makeShelfBasedAppEventDetailPage(objectGraph, container) {
    const data = dataFromDataContainer(objectGraph, container);
    if (!data) {
        return null;
    }
    const parentApp = relationshipData(objectGraph, data, "app");
    if (!parentApp) {
        return null;
    }
    const metricsOptions = {
        pageInformation: metricsPageInformationFromMediaApiResponse(objectGraph, "EventDetails", data.id, container),
        locationTracker: newLocationTracker(),
    };
    const appEvent = appEventOrPromotionStartDateFromData(objectGraph, data, parentApp, false, // `hideLockupWhenNotInstalled`
    false, // `includeClickAction`
    "light", // `offerEnvironment`
    "infer", // `offerStyle`
    false, // `includeCrossLinkStyles`
    metricsOptions, true, // `allowEndedEvents`
    true, // `includeLockupClickAction`
    null, // `editorialKind`
    false, // `isArcadePage`
    false);
    if (!appEvent || appEvent instanceof Date) {
        return null;
    }
    const appEventDetailPage = appEventDetailPageFromData(objectGraph, data, parentApp, appEvent, metricsOptions, true, // `includeLockupClickAction`
    null, // `referrerData`
    false);
    if (!appEventDetailPage) {
        return null;
    }
    const shelves = [];
    // Build the "banner" if necessary
    const bannerContext = {
        appPlatforms: supportedAppPlatformsFromData(objectGraph, data),
        offerButtonShouldBeDisabled: true,
        metricsPageInformation: metricsOptions.pageInformation,
        metricsLocationTracker: metricsOptions.locationTracker,
        webBrowser: false,
    };
    const maybeBanner = createBanner(objectGraph, parentApp, bannerContext);
    if (maybeBanner instanceof Banner) {
        shelves.push(new Shelf("banner", null, [maybeBanner]));
    }
    const appEventDetailShelf = new AppEventDetailShelf(appEventDetailPage);
    shelves.push(appEventDetailShelf);
    const page = new GenericPage(shelves);
    page.canonicalURL = attributeAsString(data, "url");
    return page;
}
//# sourceMappingURL=app-events-common.js.map