/**
 * Performs ad stitching for Search Landing page using `AdStitcher`
 */
"use strict";
import { isNothing, isSome } from "@jet/environment";
import { dataFromDataContainer } from "../../foundation/media/data-structure";
import * as adStitch from "./ad-stitcher";
// region Constants
export const searchLandingPagePositionInfo = {
    shelfIdentifier: "first",
    slot: 0,
};
/// shelf identifier for ad stitcher position info
export const searchLandingPageAdShelfIdentifier = "SLPPage";
// region Setup
/**
 * Get a positionInfo from a locationTracker and index.
 * @param locationTracker A location tracker to indicate the current parsing position.
 * @param index The index of the current position.
 * @returns A constructed `AdStitcherPositionInfo`.
 */
export function todayPositionInfoForLocationTrackerAndIndex(locationTracker, index) {
    return {
        shelfIdentifier: locationTracker.rootPosition.toString(),
        slot: index,
    };
}
/**
 * Creates an Ad stitcher for on device adverts *specifically for the product page YMAL shelf*
 * This is specific to this position as it hardcodes a shelf identifier.
 * @param adResponse Ad response to configure stitcher with.
 */
export function adStitcherForOnDeviceProductPageYMALAdvertData(objectGraph, adResponse) {
    var _a;
    const rawPositionInfo = (_a = adResponse === null || adResponse === void 0 ? void 0 : adResponse.onDeviceAd) === null || _a === void 0 ? void 0 : _a.positionInfo;
    if (isNothing(rawPositionInfo) || isNothing(adResponse)) {
        return null;
    }
    let shelfIdentifier;
    switch (adResponse.placementType) {
        case "productPageYMAL":
            shelfIdentifier = "customers-also-bought-apps";
            break;
        case "productPageYMALDuringDownload":
            shelfIdentifier = "customers-also-bought-apps-download";
            break;
        default:
            break;
    }
    if (isNothing(shelfIdentifier)) {
        return null;
    }
    // The slot as provided by ad platforms is one-based - adjust it so we're working with zero-based numbers.
    const adjustedRawSlot = rawPositionInfo.slot - 1;
    const positionInfo = {
        shelfIdentifier: shelfIdentifier,
        slot: adjustedRawSlot,
    };
    return adStitcherForOnDeviceAdvertDataAndPositionInfo(objectGraph, adResponse, positionInfo);
}
/**
 * Creates an Ad stitcher for on device adverts *specifically for search landing page*
 * This is specific to SLP because we have a hardcoded position.
 * @param adResponse Ad response to configure stitchcher with.
 * @param landingPageResponse Search page response to configure the positionInfo.
 */
export function adStitcherForOnDeviceSLPAdvertData(objectGraph, adResponse, landingPageResponse) {
    var _a;
    const adMeta = (landingPageResponse === null || landingPageResponse === void 0 ? void 0 : landingPageResponse.meta) || null;
    const slot = (_a = adMeta === null || adMeta === void 0 ? void 0 : adMeta.adDisplayStyle) === null || _a === void 0 ? void 0 : _a.slot;
    if (isSome(slot)) {
        return adStitcherForOnDeviceAdvertDataAndPositionInfo(objectGraph, adResponse, {
            shelfIdentifier: searchLandingPageAdShelfIdentifier,
            slot: slot,
        });
    }
    else {
        return adStitcherForOnDeviceAdvertDataAndPositionInfo(objectGraph, adResponse, searchLandingPagePositionInfo);
    }
}
function adStitcherForOnDeviceAdvertDataAndPositionInfo(objectGraph, adResponse, positionInfo) {
    const mediaResponse = adResponse === null || adResponse === void 0 ? void 0 : adResponse.mediaResponse;
    if (isNothing(mediaResponse) || isSome(adResponse === null || adResponse === void 0 ? void 0 : adResponse.failureReason)) {
        return null;
    }
    const stitcher = adStitch.newAdStitcher();
    /**
     * Stitch ad data to first lockup
     */
    const firstAdData = dataFromDataContainer(objectGraph, mediaResponse);
    const firstLockupAdTask = {
        data: firstAdData,
        positionInfo: positionInfo,
    };
    adStitch.addTask(stitcher, firstLockupAdTask);
    return stitcher;
}
//# sourceMappingURL=on-device-ad-stitch.js.map