/**
 * Handles stitching ads into pages.
 *
 * At a high level:
 * 1. Setup `AdStitcher` w/ content to stitch.
 * 2. Pass to builder
 * 3. Destructively consume tasks during building.
 */
"use strict";
import { isNull } from "../../foundation/json-parsing/server-data";
/**
 * Creates a string identifier representing the position information for an ad stitch task.
 * @param positionInfo The position information object to create an identifier for.
 * @returns The string identifier.
 */
function stringIdentifierForPositionInfo(positionInfo) {
    return `${positionInfo.shelfIdentifier}.${positionInfo.slot}`;
}
// region exports
/**
 * Create a new ad stitcher
 */
export function newAdStitcher() {
    return {
        tasks: {},
    };
}
/// Add task
export function addTask(stitcher, task) {
    if (isNull(stitcher)) {
        return;
    }
    const positionIdentifier = stringIdentifierForPositionInfo(task.positionInfo);
    stitcher.tasks[positionIdentifier] = task;
}
/**
 * Consume a single task, if available, for a given position information.
 * @param stitcher The relevant ad stitcher.
 * @param positionInfo The position information to check for an available task.
 * @returns A task for the position, or null if unavailable.
 */
export function consumeTask(stitcher, positionInfo) {
    if (isNull(stitcher)) {
        return null;
    }
    const positionIdentifier = stringIdentifierForPositionInfo(positionInfo);
    const task = stitcher.tasks[positionIdentifier];
    if (isNull(task)) {
        return null;
    }
    delete stitcher.tasks[positionIdentifier];
    return task;
}
/**
 * Consume all tasks for a given shelf identifier. Useful where ad data is being stitched in prior to view models being built.
 * @param stitcher The relevant ad stitcher.
 * @param shelfIdentifier The identifier for the shelf to get all available tasks for.
 * @returns An array of relevant tasks for the shelf.
 */
export function consumeTasksForShelfIdentifier(stitcher, shelfIdentifier) {
    if (isNull(stitcher)) {
        return [];
    }
    const tasksForShelf = [];
    Object.entries(stitcher.tasks).forEach(([key, value]) => {
        if (key.startsWith(shelfIdentifier)) {
            tasksForShelf.push(value);
            delete stitcher.tasks[key];
        }
    });
    return tasksForShelf;
}
//# sourceMappingURL=ad-stitcher.js.map