/**
 * Ad Incident Recorder manages the Journey and Figaro events when rendering a page.
 * This exists so managing journey and figaro metrics is a bit more sane.
 */
"use strict";
import { isNothing } from "@jet/environment";
import { DiscardAdIncident } from "../../api/models";
import { isDefinedNonNull, isNull, isNullOrEmpty } from "../../foundation/json-parsing/server-data";
import { advertInstanceIdForData } from "./ad-common";
/**
 * Create a new ad incident recorder
 */
export function newRecorder(objectGraph, iAdInfo) {
    return {
        incidents: [],
        iAdInfo: iAdInfo,
    };
}
/**
 * Get recorded incidents from recorder
 */
export function recordedIncidents(objectGraph, recorder) {
    if (isNull(recorder) || isNullOrEmpty(recorder.incidents)) {
        return null;
    }
    return recorder.incidents;
}
/**
 * Record that building a lockup from data failed.
 */
export function recordLockupFromDataFailed(objectGraph, recorder, lockupData) {
    var _a, _b;
    const instanceId = advertInstanceIdForData(objectGraph, lockupData);
    if (isNothing(instanceId)) {
        return;
    }
    const incident = objectGraph.props.enabled("advertSlotReporting")
        ? null
        : new DiscardAdIncident(instanceId, "advertDataMalformed");
    addIncident(recorder, incident);
    (_a = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _a === void 0 ? void 0 : _a.setMissedOpportunity(objectGraph, "METADATA", (_b = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _b === void 0 ? void 0 : _b.placementType);
}
/**
 * Record events that may occur when we try to fetch an on-device ad + data
 */
export function recordAdResponseEventsIfNeeded(objectGraph, recorder, response) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    if (isNullOrEmpty(response === null || response === void 0 ? void 0 : response.failureReason)) {
        return; // No failure to report
    }
    const instanceId = (_a = response === null || response === void 0 ? void 0 : response.onDeviceAd) === null || _a === void 0 ? void 0 : _a.instanceId;
    const placementType = (_b = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _b === void 0 ? void 0 : _b.placementType;
    switch (response === null || response === void 0 ? void 0 : response.failureReason) {
        case "mapiFetchFail":
            if (isDefinedNonNull(instanceId)) {
                const incident = objectGraph.props.enabled("advertSlotReporting")
                    ? null
                    : new DiscardAdIncident(instanceId, "advertDataMalformed");
                addIncident(recorder, incident);
            }
            (_c = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _c === void 0 ? void 0 : _c.setMissedOpportunity(objectGraph, "METADATA", placementType);
            break;
        case "cppAssetsMissing":
            if (isDefinedNonNull(instanceId)) {
                const incident = objectGraph.props.enabled("advertSlotReporting")
                    ? null
                    : new DiscardAdIncident(instanceId, "cppAssetsMissing");
                addIncident(recorder, incident);
            }
            (_d = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _d === void 0 ? void 0 : _d.setMissedOpportunity(objectGraph, "METADATA", placementType);
            break;
        case "insufficientAssets":
            if (isDefinedNonNull(instanceId)) {
                const incident = objectGraph.props.enabled("advertSlotReporting")
                    ? null
                    : new DiscardAdIncident(instanceId, "insufficientAssets");
                addIncident(recorder, incident);
            }
            (_e = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _e === void 0 ? void 0 : _e.setMissedOpportunity(objectGraph, "METADATA", placementType);
            break;
        case "noAdAvailable":
            // no journey metric
            (_f = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _f === void 0 ? void 0 : _f.setMissedOpportunity(objectGraph, "NOAD", placementType);
            break;
        case "timeout":
            // no journey metric
            (_g = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _g === void 0 ? void 0 : _g.setMissedOpportunity(objectGraph, "APSLA", placementType);
            break;
        case "localizationNotAvailable":
            if (isDefinedNonNull(instanceId)) {
                const incident = objectGraph.props.enabled("advertSlotReporting")
                    ? null
                    : new DiscardAdIncident(instanceId, "localizationNotAvailable");
                addIncident(recorder, incident);
            }
            (_h = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _h === void 0 ? void 0 : _h.setMissedOpportunity(objectGraph, "NOLOC", placementType);
            break;
        case "policyAdDrop":
            // no journey metric
            (_j = recorder === null || recorder === void 0 ? void 0 : recorder.iAdInfo) === null || _j === void 0 ? void 0 : _j.setMissedOpportunity(objectGraph, "ODP_NOAD", placementType);
            break;
        default:
            break;
    }
}
// region Internals
/**
 * Add incident on `AdIncidentRecorder`
 */
function addIncident(recorder, incident) {
    if (isNull(recorder) || isNull(incident)) {
        return;
    }
    recorder.incidents.push(incident);
}
//# sourceMappingURL=ad-incident-recorder.js.map